<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Autograph Feedback Insights Controller
 * Handles all admin panel interactions
 */
class Autograph_feedback_insights extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('autograph_feedback_insights/autograph_feedback_insights_model');
    }

    /**
     * Main insights page
     */
    public function index()
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        // Handle manual analysis trigger
        if ($this->input->post('run_analysis')) {
            $this->handle_manual_analysis();
        }

        // Handle settings update
        if ($this->input->post('update_settings')) {
            $this->handle_settings_update();
        }

        // Load current settings
        $data['settings'] = $this->autograph_feedback_insights_model->get_all_settings();
        
        // Load latest report
        $data['latest_report'] = $this->autograph_feedback_insights_model->get_latest_report();
        
        // Load all reports for history
        $data['reports'] = $this->autograph_feedback_insights_model->get_all_reports(20);
        
        // Get lead statuses for filter
        $this->load->model('leads_model');
        $data['lead_statuses'] = $this->leads_model->get_status();
        
        // Get notes count with current filter
        $status_filter_setting = $data['settings']['status_filter'] ?? '';
        $status_filter = [];
        if (!empty($status_filter_setting)) {
            $status_filter = json_decode($status_filter_setting, true);
            if (!is_array($status_filter)) {
                $status_filter = [];
            }
        }
        $notes = $this->autograph_feedback_insights_model->get_autograph_notes($status_filter);
        $data['notes_count'] = count($notes);
        
        $data['title'] = _l('autograph_insights');
        
        $this->load->view('insights', $data);
    }

    /**
     * Manage automated schedules
     */
    public function schedules()
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        // Handle schedule creation/update
        if ($this->input->post('save_schedule')) {
            $this->handle_schedule_save();
        }

        // Handle schedule deletion
        if ($this->input->post('delete_schedule')) {
            $schedule_id = $this->input->post('schedule_id');
            $this->autograph_feedback_insights_model->delete_schedule($schedule_id);
            set_alert('success', 'Schedule deleted successfully');
            redirect(admin_url('autograph_feedback_insights/schedules'));
        }

        // Get lead statuses for filter
        $this->load->model('leads_model');
        $data['lead_statuses'] = $this->leads_model->get_status();

        // Get all schedules
        $data['schedules'] = $this->autograph_feedback_insights_model->get_all_schedules();
        
        $data['title'] = 'Automated Report Schedules';
        
        $this->load->view('schedules', $data);
    }

    /**
     * Handle schedule save
     */
    private function handle_schedule_save()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        $schedule_id = $this->input->post('schedule_id');
        
        // Prepare status filter
        $status_filter = $this->input->post('status_filter');
        if (is_array($status_filter) && !empty($status_filter)) {
            $status_filter = json_encode($status_filter);
        } else {
            $status_filter = '';
        }

        // Prepare WhatsApp numbers
        $whatsapp_numbers_raw = $this->input->post('whatsapp_numbers');
        $whatsapp_numbers = array_filter(array_map('trim', explode("\n", $whatsapp_numbers_raw)));
        $whatsapp_numbers = json_encode($whatsapp_numbers);

        $data = [
            'schedule_name' => $this->input->post('schedule_name'),
            'status_filter' => $status_filter,
            'date_range_type' => $this->input->post('date_range_type'),
            'custom_start_date' => $this->input->post('custom_start_date'),
            'custom_end_date' => $this->input->post('custom_end_date'),
            'schedule_time' => $this->input->post('schedule_time'),
            'schedule_frequency' => $this->input->post('schedule_frequency'),
            'schedule_day' => $this->input->post('schedule_day'),
            'whatsapp_numbers' => $whatsapp_numbers,
            'active' => $this->input->post('active') ? 1 : 0,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        // Calculate next run time
        $data['next_run'] = $this->calculate_next_run($data);

        if (empty($schedule_id)) {
            // Create new schedule
            $data['created_at'] = date('Y-m-d H:i:s');
            $this->autograph_feedback_insights_model->create_schedule($data);
            set_alert('success', 'Schedule created successfully');
        } else {
            // Update existing schedule
            $this->autograph_feedback_insights_model->update_schedule($schedule_id, $data);
            set_alert('success', 'Schedule updated successfully');
        }

        redirect(admin_url('autograph_feedback_insights/schedules'));
    }

    /**
     * Calculate next run time based on frequency
     */
    private function calculate_next_run($schedule_data)
    {
        $time = $schedule_data['schedule_time'];
        $frequency = $schedule_data['schedule_frequency'];
        $day = $schedule_data['schedule_day'];

        $now = time();
        $today = date('Y-m-d');
        $schedule_datetime = strtotime($today . ' ' . $time);

        switch ($frequency) {
            case 'daily':
                // If time has passed today, schedule for tomorrow
                if ($schedule_datetime <= $now) {
                    $next_run = date('Y-m-d H:i:s', strtotime('+1 day', $schedule_datetime));
                } else {
                    $next_run = date('Y-m-d H:i:s', $schedule_datetime);
                }
                break;

            case 'weekly':
                // Find next occurrence of the specified day
                $current_day = date('N'); // 1 (Monday) - 7 (Sunday)
                $target_day = $day ?: 1;
                
                if ($target_day > $current_day || ($target_day == $current_day && $schedule_datetime > $now)) {
                    $days_ahead = $target_day - $current_day;
                } else {
                    $days_ahead = 7 - $current_day + $target_day;
                }
                
                $next_run = date('Y-m-d H:i:s', strtotime("+{$days_ahead} days", $schedule_datetime));
                break;

            case 'monthly':
                // Find next occurrence of the specified day of month
                $current_day = date('j');
                $target_day = $day ?: 1;
                
                if ($target_day > $current_day || ($target_day == $current_day && $schedule_datetime > $now)) {
                    // This month
                    $next_run = date('Y-m-d H:i:s', strtotime(date('Y-m-') . $target_day . ' ' . $time));
                } else {
                    // Next month
                    $next_run = date('Y-m-d H:i:s', strtotime('next month', strtotime(date('Y-m-') . $target_day . ' ' . $time)));
                }
                break;

            default:
                $next_run = date('Y-m-d H:i:s', strtotime('+1 day', $schedule_datetime));
        }

        return $next_run;
    }

    /**
     * Get schedule for editing (AJAX)
     */
    public function get_schedule($schedule_id)
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $schedule = $this->autograph_feedback_insights_model->get_schedule($schedule_id);
        
        if ($schedule) {
            // Decode JSON fields
            $schedule->status_filter = json_decode($schedule->status_filter, true) ?: [];
            $schedule->whatsapp_numbers = json_decode($schedule->whatsapp_numbers, true) ?: [];
            $schedule->whatsapp_numbers_text = implode("\n", $schedule->whatsapp_numbers);
            
            echo json_encode([
                'success' => true,
                'schedule' => $schedule
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Schedule not found'
            ]);
        }
    }

    /**
     * Handle manual analysis trigger
     */
    private function handle_manual_analysis()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        $result = $this->autograph_feedback_insights_model->run_analysis();
        
        if ($result['success']) {
            set_alert('success', sprintf(
                _l('autograph_analysis_success'),
                $result['notes_analyzed'],
                $result['leads_analyzed']
            ));
        } else {
            set_alert('danger', $result['message']);
        }
        
        redirect(admin_url('autograph_feedback_insights'));
    }

    /**
     * Handle settings form submission
     */
    private function handle_settings_update()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        $status_filter = $this->input->post('status_filter');
        if (is_array($status_filter)) {
            $status_filter = json_encode($status_filter);
        } else {
            $status_filter = '';
        }
        
        $settings = [
            'openai_api_key' => $this->input->post('openai_api_key'),
            'openai_model' => $this->input->post('openai_model') ?: 'gpt-4',
            'cron_enabled' => $this->input->post('cron_enabled') ? '1' : '0',
            'status_filter' => $status_filter,
            'whatsapp_phone' => $this->input->post('whatsapp_phone'),
        ];

        foreach ($settings as $name => $value) {
            $this->autograph_feedback_insights_model->update_setting($name, $value);
        }

        set_alert('success', _l('autograph_settings_saved'));
        redirect(admin_url('autograph_feedback_insights'));
    }

    /**
     * AJAX endpoint to run analysis
     */
    public function run_analysis_ajax()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $result = $this->autograph_feedback_insights_model->run_analysis();
        
        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => sprintf(
                    _l('autograph_analysis_success'),
                    $result['notes_analyzed'],
                    $result['leads_analyzed']
                ),
                'notes_analyzed' => $result['notes_analyzed'],
                'leads_analyzed' => $result['leads_analyzed']
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => $result['message']
            ]);
        }
    }

    /**
     * AJAX endpoint to get latest report
     */
    public function get_latest_report_ajax()
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $report = $this->autograph_feedback_insights_model->get_latest_report();
        
        if ($report) {
            // Decode JSON fields
            $report->positive_highlights = json_decode($report->positive_highlights, true) ?: [];
            $report->negative_concerns = json_decode($report->negative_concerns, true) ?: [];
            $report->action_suggestions = json_decode($report->action_suggestions, true) ?: [];
            
            echo json_encode([
                'success' => true,
                'report' => $report
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'No report found'
            ]);
        }
    }

    /**
     * View a specific report
     */
    public function view_report($report_id)
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        $report = $this->autograph_feedback_insights_model->get_report($report_id);
        
        if (!$report) {
            set_alert('danger', 'Report not found');
            redirect(admin_url('autograph_feedback_insights'));
        }

        $data['report'] = $report;
        $data['title'] = 'Autograph Insights Report - ' . date('Y-m-d H:i', strtotime($report->analysis_date));
        
        $this->load->view('view_report', $data);
    }

    /**
     * Export report to PDF
     */
    public function export_pdf($report_id)
    {
        if (!has_permission('autograph_feedback_insights', '', 'view') && !is_admin()) {
            access_denied('autograph_feedback_insights');
        }

        $report = $this->autograph_feedback_insights_model->get_report($report_id);
        
        if (!$report) {
            set_alert('danger', 'Report not found');
            redirect(admin_url('autograph_feedback_insights'));
        }

        // Decode JSON fields
        $positive_highlights = json_decode($report->positive_highlights, true) ?: [];
        $negative_concerns = json_decode($report->negative_concerns, true) ?: [];
        $action_suggestions = json_decode($report->action_suggestions, true) ?: [];

        // Create PDF using TCPDF directly
        require_once(APPPATH . 'vendor/tecnickcom/tcpdf/tcpdf.php');
        
        $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
        
        // Set document information
        $pdf->SetCreator('Perfex CRM');
        $pdf->SetAuthor('Autograph Feedback Insights');
        $pdf->SetTitle('Sentiment Analysis Report - ' . date('Y-m-d', strtotime($report->analysis_date)));
        
        // Remove default header/footer
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);
        
        // Set margins
        $pdf->SetMargins(15, 15, 15);
        $pdf->SetAutoPageBreak(true, 15);
        
        // Add a page
        $pdf->AddPage();
        
        // Set font
        $pdf->SetFont('helvetica', 'B', 20);
        
        // Title
        $pdf->Cell(0, 10, 'Autograph Feedback Insights', 0, 1, 'C');
        $pdf->SetFont('helvetica', '', 12);
        $pdf->Cell(0, 10, 'Sentiment Analysis Report', 0, 1, 'C');
        $pdf->Ln(5);
        
        // Report Info
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(0, 6, 'Analysis Date: ' . date('F j, Y g:i A', strtotime($report->analysis_date)), 0, 1);
        $pdf->Cell(0, 6, 'Notes Analyzed: ' . $report->notes_analyzed, 0, 1);
        $pdf->Cell(0, 6, 'Leads Analyzed: ' . $report->leads_analyzed, 0, 1);
        $pdf->Ln(5);
        
        // Overall Sentiment
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->Cell(0, 8, 'Overall Sentiment', 0, 1);
        $pdf->SetFont('helvetica', '', 11);
        
        $sentiment_color = [200, 200, 200];
        if ($report->sentiment == 'positive') {
            $sentiment_color = [76, 175, 80];
        } elseif ($report->sentiment == 'negative') {
            $sentiment_color = [244, 67, 54];
        }
        
        $pdf->SetFillColor($sentiment_color[0], $sentiment_color[1], $sentiment_color[2]);
        $pdf->SetTextColor(255, 255, 255);
        $pdf->Cell(40, 8, ucfirst($report->sentiment), 0, 0, 'C', true);
        
        if ($report->sentiment_score) {
            $pdf->SetTextColor(0, 0, 0);
            $pdf->Cell(60, 8, ' Score: ' . number_format($report->sentiment_score * 100, 1) . '%', 0, 1);
        } else {
            $pdf->Ln();
        }
        
        $pdf->SetTextColor(0, 0, 0);
        $pdf->Ln(5);
        
        // Summary
        if (!empty($report->summary)) {
            $pdf->SetFont('helvetica', 'B', 14);
            $pdf->Cell(0, 8, 'Summary', 0, 1);
            $pdf->SetFont('helvetica', '', 10);
            $pdf->MultiCell(0, 6, $report->summary, 0, 'L');
            $pdf->Ln(3);
        }
        
        // Positive Highlights
        if (!empty($positive_highlights)) {
            $pdf->SetFont('helvetica', 'B', 14);
            $pdf->SetTextColor(76, 175, 80);
            $pdf->Cell(0, 8, 'Key Positive Highlights', 0, 1);
            $pdf->SetTextColor(0, 0, 0);
            $pdf->SetFont('helvetica', '', 10);
            
            foreach ($positive_highlights as $highlight) {
                $pdf->MultiCell(0, 6, '• ' . $highlight, 0, 'L');
            }
            $pdf->Ln(3);
        }
        
        // Negative Concerns
        if (!empty($negative_concerns)) {
            $pdf->SetFont('helvetica', 'B', 14);
            $pdf->SetTextColor(244, 67, 54);
            $pdf->Cell(0, 8, 'Key Concerns / Objections', 0, 1);
            $pdf->SetTextColor(0, 0, 0);
            $pdf->SetFont('helvetica', '', 10);
            
            foreach ($negative_concerns as $concern) {
                $pdf->MultiCell(0, 6, '• ' . $concern, 0, 'L');
            }
            $pdf->Ln(3);
        }
        
        // Market Narrative
        if (!empty($report->market_narrative)) {
            $pdf->SetFont('helvetica', 'B', 14);
            $pdf->Cell(0, 8, 'Market Narrative', 0, 1);
            $pdf->SetFont('helvetica', '', 10);
            $pdf->MultiCell(0, 6, $report->market_narrative, 0, 'L');
            $pdf->Ln(3);
        }
        
        // Action Suggestions
        if (!empty($action_suggestions)) {
            $pdf->SetFont('helvetica', 'B', 14);
            $pdf->SetTextColor(33, 150, 243);
            $pdf->Cell(0, 8, 'Action Suggestions', 0, 1);
            $pdf->SetTextColor(0, 0, 0);
            $pdf->SetFont('helvetica', '', 10);
            
            foreach ($action_suggestions as $suggestion) {
                $pdf->MultiCell(0, 6, '• ' . $suggestion, 0, 'L');
            }
        }
        
        // Output PDF
        $filename = 'Autograph_Insights_Report_' . date('Y-m-d', strtotime($report->analysis_date)) . '.pdf';
        $pdf->Output($filename, 'D');
    }

    /**
     * Send report to WhatsApp (legacy - redirects to Dashboard)
     */
    public function send_to_whatsapp($report_id)
    {
        // Redirect back to dashboard where they can use the new multi-select modal
        redirect(admin_url('autograph_feedback_insights#report-' . $report_id));
    }

    /**
     * Send report to multiple WhatsApp numbers (AJAX)
     */
    public function send_to_whatsapp_multiple()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $report_id = $this->input->post('report_id');
        $phone_numbers = $this->input->post('phone_numbers');

        if (empty($report_id) || empty($phone_numbers) || !is_array($phone_numbers)) {
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
            return;
        }

        $report = $this->autograph_feedback_insights_model->get_report($report_id);
        
        if (!$report) {
            echo json_encode(['success' => false, 'message' => 'Report not found']);
            return;
        }

        // Send to each number
        $sent_success = 0;
        $sent_failed = 0;
        $errors = [];

        foreach ($phone_numbers as $phone) {
            $phone = trim($phone);
            if (empty($phone)) {
                continue;
            }

            $result = $this->autograph_feedback_insights_model->send_report_to_whatsapp($report, $phone);
            
            if ($result['success']) {
                $sent_success++;
            } else {
                $sent_failed++;
                $errors[] = $phone . ': ' . ($result['error'] ?? 'Unknown error');
            }
        }

        if ($sent_success > 0 && $sent_failed == 0) {
            echo json_encode([
                'success' => true,
                'message' => "Report sent successfully to {$sent_success} number(s)!"
            ]);
        } elseif ($sent_success > 0 && $sent_failed > 0) {
            echo json_encode([
                'success' => true,
                'message' => "Sent to {$sent_success} number(s). Failed: {$sent_failed}. Errors: " . implode(', ', $errors)
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => "Failed to send to all numbers. Errors: " . implode(', ', $errors)
            ]);
        }
    }

    /**
     * Run analysis and auto-send to configured WhatsApp numbers
     */
    public function run_analysis_and_send()
    {
        if (!has_permission('autograph_feedback_insights', '', 'edit') && !is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        // Run analysis first
        $result = $this->autograph_feedback_insights_model->run_analysis();
        
        if (!$result['success']) {
            echo json_encode($result);
            return;
        }

        // Get WhatsApp numbers from settings
        $whatsapp_phone_setting = $this->autograph_feedback_insights_model->get_setting('whatsapp_phone');
        $whatsapp_numbers = array_filter(array_map('trim', explode("\n", $whatsapp_phone_setting)));

        if (empty($whatsapp_numbers)) {
            // No WhatsApp numbers configured, just return analysis result
            echo json_encode($result);
            return;
        }

        // Get the report
        $report = $this->autograph_feedback_insights_model->get_report($result['report_id']);

        // Send to all configured numbers
        $sent_success = 0;
        $sent_failed = 0;

        foreach ($whatsapp_numbers as $phone) {
            $send_result = $this->autograph_feedback_insights_model->send_report_to_whatsapp($report, $phone);
            if ($send_result['success']) {
                $sent_success++;
            } else {
                $sent_failed++;
            }
        }

        $result['whatsapp_sent'] = $sent_success;
        $result['whatsapp_failed'] = $sent_failed;
        
        if ($sent_success > 0) {
            $result['message'] .= " Report sent to {$sent_success} WhatsApp number(s).";
        }

        echo json_encode($result);
    }

}

