<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Leads AI Analysis Model
 * 
 * Handles AI-driven lead evaluation reports and analysis
 * 
 * @package    Application
 * @subpackage Models
 * @category   Lead Management
 * @author     Ibraya Group
 * @since      Version 1.0.0
 */
class Leads_ai_model extends App_Model
{
    /**
     * Class constructor
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get the latest AI report for a lead
     * 
     * @param  int $lead_id Lead ID
     * @return object|null AI report data or null if not found
     */
    public function get_latest_report($lead_id)
    {
        return $this->db->where('lead_id', $lead_id)
                       ->order_by('created_at', 'DESC')
                       ->limit(1)
                       ->get(db_prefix() . 'leads_ai_reports')
                       ->row();
    }

    /**
     * Get all AI reports for a lead
     * 
     * @param  int $lead_id Lead ID
     * @return array Array of AI reports
     */
    public function get_reports($lead_id)
    {
        return $this->db->where('lead_id', $lead_id)
                       ->order_by('created_at', 'DESC')
                       ->get(db_prefix() . 'leads_ai_reports')
                       ->result();
    }

    /**
     * Save AI analysis report
     * 
     * @param  int    $lead_id        Lead ID
     * @param  int    $score          Analysis score (0-100)
     * @param  string $verdict        Verdict (Likely/Unlikely)
     * @param  string $rationale      Analysis rationale
     * @param  string $recommendations Recommendations
     * @param  array  $uploaded_files  Array of uploaded file names
n     * @return int|false Insert ID on success, false on failure
     */
    public function save_report($lead_id, $score, $verdict, $rationale, $recommendations = '', $uploaded_files = [], $extra_data = [])
    {
        $data = [
            'lead_id'             => $lead_id,
            'analysis_type'       => 'enhanced_conversation_analysis',
            'confidence_score'    => $score,
            'purchase_likelihood' => $verdict,
            'key_insights'        => $rationale,
            'recommendations'     => $recommendations,
            'uploaded_files'      => json_encode($uploaded_files),
            'status'              => 'completed',
            'created_by'          => get_staff_user_id(),
            'created_at'          => date('Y-m-d H:i:s'),
        ];
        
        // Add extra analysis data as JSON in ai_response field
        if (!empty($extra_data)) {
            $data['ai_response'] = json_encode($extra_data);
        }

        $this->db->insert(db_prefix() . 'leads_ai_reports', $data);
        
        if ($this->db->affected_rows() > 0) {
            $insert_id = $this->db->insert_id();
            
            // Log activity
            $this->log_ai_analysis_activity($lead_id, $score, $verdict);
            
            return $insert_id;
        }
        
        return false;
    }

    /**
     * Delete AI report
     * 
     * @param  int $report_id Report ID
     * @return bool True on success, false on failure
     */
    public function delete_report($report_id)
    {
        $this->db->where('id', $report_id);
        return $this->db->delete(db_prefix() . 'leads_ai_reports');
    }

    /**
     * Check if lead has any AI reports
     * 
     * @param  int $lead_id Lead ID
     * @return bool True if reports exist, false otherwise
     */
    public function has_reports($lead_id)
    {
        return $this->db->where('lead_id', $lead_id)
                       ->count_all_results(db_prefix() . 'leads_ai_reports') > 0;
    }

    /**
     * Get AI analysis statistics
     * 
     * @return array Statistics data
     */
    public function get_analysis_stats()
    {
        // Total analyses
        $total = $this->db->count_all_results(db_prefix() . 'leads_ai_reports');
        
        // Likely vs Unlikely
        $likely = $this->db->where('verdict', 'Likely')
                          ->count_all_results(db_prefix() . 'leads_ai_reports');
        
        $unlikely = $this->db->where('verdict', 'Unlikely')
                            ->count_all_results(db_prefix() . 'leads_ai_reports');
        
        // Average score
        $avg_score = $this->db->select_avg('score')
                             ->get(db_prefix() . 'leads_ai_reports')
                             ->row()->score ?? 0;

        return [
            'total_analyses' => $total,
            'likely_count'   => $likely,
            'unlikely_count' => $unlikely,
            'average_score'  => round($avg_score, 1),
            'success_rate'   => $total > 0 ? round(($likely / $total) * 100, 1) : 0
        ];
    }

    /**
     * Log AI analysis activity
     * 
     * @param  int    $lead_id Lead ID
     * @param  int    $score   Analysis score
     * @param  string $verdict Analysis verdict
     * @return bool   True on success
     */
    private function log_ai_analysis_activity($lead_id, $score, $verdict)
    {
        return $this->leads_model->log_lead_activity($lead_id, 
            'lead_activity_ai_analysis_completed', 
            false, 
            serialize([
                'score'   => $score,
                'verdict' => $verdict
            ])
        );
    }

    /**
     * Clean up old AI analysis files
     * 
     * @param  int $days_old Files older than this many days will be deleted
     * @return int Number of files cleaned up
     */
    public function cleanup_old_files($days_old = 30)
    {
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));
        
        // Get reports older than cutoff date
        $old_reports = $this->db->where('created_at <', $cutoff_date)
                               ->get(db_prefix() . 'leads_ai_reports')
                               ->result();
        
        $cleaned_count = 0;
        
        foreach ($old_reports as $report) {
            $uploaded_files = json_decode($report->uploaded_files, true);
            
            if (is_array($uploaded_files)) {
                foreach ($uploaded_files as $file) {
                    $file_path = get_upload_path_by_type('lead_ai') . $report->lead_id . '/' . $file;
                    if (file_exists($file_path)) {
                        unlink($file_path);
                        $cleaned_count++;
                    }
                }
            }
            
            // Delete the report record
            $this->db->where('id', $report->id);
            $this->db->delete(db_prefix() . 'leads_ai_reports');
        }
        
        return $cleaned_count;
    }

    /**
     * Validate uploaded files for AI analysis (updated for modern OpenAI API)
     * 
     * @param  array $files Array of uploaded files
     * @return array Validation result ['valid' => bool, 'errors' => array]
     */
    public function validate_upload_files($files)
    {
        $errors = [];
        
        // Updated file type support based on modern OpenAI API capabilities
        $image_types = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp'];
        $document_types = ['pdf', 'doc', 'docx', 'txt', 'csv', 'xls', 'xlsx'];
        $allowed_types = array_merge($image_types, $document_types);
        
        $max_files = 10;
        $max_image_size = 20 * 1024 * 1024; // 20MB for images (OpenAI limit)
        $max_pdf_size = 32 * 1024 * 1024; // 32MB for PDFs (OpenAI limit)
        $max_doc_size = 10 * 1024 * 1024; // 10MB for other documents

        if (empty($files)) {
            $errors[] = _l('ai_analysis_no_files');
        }

        if (count($files) > $max_files) {
            $errors[] = "Maximum {$max_files} files allowed";
        }

        foreach ($files as $file) {
            // Check file type
            $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            if (!in_array($file_ext, $allowed_types)) {
                $errors[] = "File {$file['name']} has invalid type. Allowed: " . implode(', ', $allowed_types);
                continue;
            }

            // Check file size based on type
            $max_size = $max_doc_size; // Default
            
            if (in_array($file_ext, $image_types)) {
                $max_size = $max_image_size;
                $type_desc = "image";
            } elseif ($file_ext === 'pdf') {
                $max_size = $max_pdf_size;
                $type_desc = "PDF";
            } else {
                $type_desc = "document";
            }
            
            if ($file['size'] > $max_size) {
                $size_mb = round($max_size / (1024 * 1024));
                $errors[] = "File {$file['name']} is too large. Maximum {$size_mb}MB for {$type_desc} files.";
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    /**
     * Log API call with error handling and reconnection
     * 
     * @param  array $log_data API call data to log
     * @return bool Success status
     */
    public function log_api_call($log_data)
    {
        try {
            $table = db_prefix() . 'ai_api_logs';
            
            // Create table if it doesn't exist
            if (!$this->db->table_exists($table)) {
                $this->_create_api_logs_table();
            }
            
            $default_data = [
                'lead_id' => null,
                'endpoint' => '',
                'method' => 'POST',
                'request_data' => '',
                'response_data' => '',
                'status_code' => null,
                'response_time' => null,
                'success' => 0,
                'error_message' => null,
                'staff_id' => get_staff_user_id(),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $log_data = array_merge($default_data, $log_data);
            
            // Truncate large data to prevent MySQL packet size issues
            if (isset($log_data['request_data']) && strlen($log_data['request_data']) > 50000) {
                $log_data['request_data'] = substr($log_data['request_data'], 0, 50000) . '... [TRUNCATED]';
            }
            if (isset($log_data['response_data']) && strlen($log_data['response_data']) > 50000) {
                $log_data['response_data'] = substr($log_data['response_data'], 0, 50000) . '... [TRUNCATED]';
            }
            
            return $this->db->insert($table, $log_data);
            
        } catch (Exception $e) {
            // Handle MySQL connection issues
            if (strpos($e->getMessage(), 'MySQL server has gone away') !== false) {
                log_message('info', 'MySQL connection lost during API logging, attempting to reconnect...');
                try {
                    $this->db->reconnect();
                    return $this->db->insert($table, $log_data);
                } catch (Exception $reconnect_error) {
                    log_message('error', 'Failed to reconnect and log API call: ' . $reconnect_error->getMessage());
                    return false;
                }
            } else {
                log_message('error', 'Failed to log API call: ' . $e->getMessage());
                return false;
            }
        }
    }

    /**
     * Create API logs table
     * 
     * @return void
     */
    private function _create_api_logs_table()
    {
        $table = db_prefix() . 'ai_api_logs';
        
        $sql = "CREATE TABLE IF NOT EXISTS `{$table}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `lead_id` int(11) DEFAULT NULL,
            `endpoint` varchar(255) NOT NULL,
            `method` varchar(10) NOT NULL DEFAULT 'POST',
            `request_data` longtext,
            `response_data` longtext,
            `status_code` int(3) DEFAULT NULL,
            `response_time` float DEFAULT NULL,
            `success` tinyint(1) DEFAULT 0,
            `error_message` text,
            `staff_id` int(11) DEFAULT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `lead_id` (`lead_id`),
            KEY `status_code` (`status_code`),
            KEY `success` (`success`),
            KEY `created_at` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
        
        $this->db->query($sql);
    }

    /**
     * Parse and extract content from uploaded files for AI analysis
     * 
     * @param  int   $lead_id     Lead ID
     * @param  array $file_paths  Array of file paths to parse
     * @return array             Array of parsed file contents
     */
    public function parse_uploaded_files($lead_id, $file_paths = [])
    {
        $this->load->helper('ai_file_parser');
        
        $parsed_files = [];
        
        if (empty($file_paths)) {
            // Get uploaded files for this lead from database or directory
            $file_paths = $this->get_lead_uploaded_files($lead_id);
        }
        
        foreach ($file_paths as $file_path) {
            $filename = basename($file_path);
            
            if (!file_exists($file_path)) {
                $parsed_files[$filename] = [
                    'success' => false,
                    'error' => 'File not found: ' . $file_path,
                    'content' => '',
                    'metadata' => []
                ];
                continue;
            }
            
            // Get file MIME type
            $file_type = mime_content_type($file_path);
            if (!$file_type) {
                $file_type = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            }
            
            // Parse the file
            $parsed_result = parse_file_for_ai_analysis($file_path, $file_type);
            $parsed_files[$filename] = $parsed_result;
            
            // Log parsing activity
            log_message('info', "Parsed file for lead {$lead_id}: {$filename} - Success: " . ($parsed_result['success'] ? 'Yes' : 'No'));
        }
        
        return $parsed_files;
    }

    /**
     * Get uploaded files for a lead
     * 
     * @param  int $lead_id Lead ID
     * @return array        Array of file paths
     */
    public function get_lead_uploaded_files($lead_id)
    {
        $files = [];
        
        // Check for files in upload directory
        $upload_path = FCPATH . 'uploads/ai_analysis/lead_' . $lead_id . '/';
        
        if (is_dir($upload_path)) {
            $file_list = scandir($upload_path);
            foreach ($file_list as $file) {
                if ($file !== '.' && $file !== '..' && is_file($upload_path . $file)) {
                    $files[] = $upload_path . $file;
                }
            }
        }
        
        return $files;
    }

    /**
     * Enhanced AI analysis with modern OpenAI API integration
     * 
     * @param  int    $lead_id Lead ID
     * @param  array  $lead_data Lead data from database
     * @param  array  $uploaded_files Optional array of file paths
     * @return array  Analysis result
     */
    public function analyze_lead_with_files($lead_id, $lead_data, $uploaded_files = [])
    {
        try {
            // Parse uploaded files and prepare for modern OpenAI API
            $parsed_files = $this->parse_uploaded_files($lead_id, $uploaded_files);
            
            // Prepare content array for multimodal API
            $content_array = $this->prepare_multimodal_content($lead_data, $parsed_files, $uploaded_files);
            
            // Get AI configuration
            $model = get_option('ai_model') ?: 'gpt-4o';
            $temperature = get_option('ai_temperature') ?: 0.3;
            $max_tokens = get_option('ai_max_tokens') ?: 1000;
            
            // Make enhanced API call to OpenAI
            $api_result = $this->make_enhanced_openai_api_call($content_array, $model, $temperature, $max_tokens);
            
            // If enhanced call fails, try fallback to text-only
            if (!$api_result['success']) {
                error_log('Multimodal API call failed, trying text-only fallback: ' . $api_result['error']);
                
                // Build text-only prompt with file content included - use sanitized version
                $text_prompt = $this->build_structured_analysis_prompt($lead_data, $parsed_files);
                
                // Additional sanitization for problematic content
                $text_prompt = $this->sanitize_text($text_prompt);
                
                $api_result = $this->make_openai_api_call($text_prompt, $model, $temperature, $max_tokens);
                
                // If text-only also fails, try a simplified version
                if (!$api_result['success']) {
                    error_log('Text-only API call also failed, trying simplified prompt: ' . $api_result['error']);
                    $simplified_prompt = $this->build_simplified_prompt($lead_data);
                    $api_result = $this->make_openai_api_call($simplified_prompt, $model, $temperature, $max_tokens);
                }
            }
            
            // Log API call
            $this->log_api_call([
                'lead_id' => $lead_id,
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'request_data' => json_encode([
                    'model' => $model,
                    'temperature' => $temperature,
                    'max_tokens' => $max_tokens,
                    'content_items' => count($content_array),
                    'files_count' => count($parsed_files)
                ]),
                'response_data' => json_encode($api_result),
                'status_code' => $api_result['success'] ? 200 : 400,
                'response_time' => $api_result['response_time'] ?? null,
                'success' => $api_result['success'] ? 1 : 0,
                'error_message' => $api_result['success'] ? null : $api_result['error']
            ]);
            
            if (!$api_result['success']) {
                return [
                    'success' => false,
                    'error' => $api_result['error']
                ];
            }
            
            // Parse AI response (now expecting structured JSON)
            $analysis = $this->parse_structured_ai_response($api_result['response']);
            
            // Save report with file information and enhanced data
            $uploaded_file_names = array_keys($parsed_files);
            $report_id = $this->save_report(
                $lead_id,
                $analysis['score'],
                $analysis['verdict'],
                $analysis['rationale'],
                $analysis['recommendations'],
                $uploaded_file_names,
                [
                    'communication_analysis' => $analysis['communication_analysis'] ?? '',
                    'confidence_level' => $analysis['confidence_level'] ?? 'Medium',
                    'risk_factors' => $analysis['risk_factors'] ?? [],
                    'positive_signals' => $analysis['positive_signals'] ?? [],
                    'key_insights_array' => $analysis['key_insights'] ?? []
                ]
            );
            
            return [
                'success' => true,
                'report_id' => $report_id,
                'analysis' => $analysis,
                'parsed_files' => $parsed_files
            ];
            
        } catch (Exception $e) {
            log_message('error', 'AI analysis with files failed: ' . $e->getMessage());
            
            // Log failed API call
            $this->log_api_call([
                'lead_id' => $lead_id,
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'request_data' => 'Error before API call',
                'response_data' => '',
                'status_code' => 0,
                'success' => 0,
                'error_message' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'error' => 'Analysis failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Prepare multimodal content array for modern OpenAI API
     * 
     * @param  array $lead_data Lead data
     * @param  array $parsed_files Parsed file contents
     * @param  array $file_paths Original file paths
     * @return array Content array for OpenAI API
     */
    private function prepare_multimodal_content($lead_data, $parsed_files, $file_paths)
    {
        $content = [];
        
        // Start with structured analysis prompt
        $prompt_text = $this->build_structured_analysis_prompt($lead_data, $parsed_files);
        
        // Validate prompt text before adding to content
        if (!mb_check_encoding($prompt_text, 'UTF-8')) {
            error_log('Prompt text contains invalid UTF-8 characters, fixing...');
            $prompt_text = mb_convert_encoding($prompt_text, 'UTF-8', 'UTF-8');
        }
        
        $content[] = [
            'type' => 'text',
            'text' => $prompt_text
        ];
        
        $images_added = 0;
        $max_images = 3; // Reduced to prevent payload issues
        
        // Only try to add small images to avoid JSON issues
        foreach ($file_paths as $file_path) {
            if (!file_exists($file_path) || $images_added >= $max_images) {
                continue;
            }
            
            $file_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            $filename = basename($file_path);
            
            // Handle only small images with base64 encoding
            if (in_array($file_ext, ['jpg', 'jpeg', 'png'])) {
                try {
                    $file_size = filesize($file_path);
                    
                    // Only process small images to avoid JSON issues
                    if ($file_size > 2 * 1024 * 1024) { // 2MB limit for stability
                        error_log("Image {$filename} is too large ({$file_size} bytes), skipping for multimodal");
                        continue;
                    }
                    
                    $image_data = file_get_contents($file_path);
                    if ($image_data === false) {
                        error_log("Could not read image file: {$filename}");
                        continue;
                    }
                    
                    $base64_image = base64_encode($image_data);
                    if ($base64_image === false) {
                        error_log("Failed to base64 encode image: {$filename}");
                        continue;
                    }
                    
                    // Use simple MIME type
                    $mime_type = ($file_ext === 'png') ? 'image/png' : 'image/jpeg';
                    
                    $image_url_data = [
                        'type' => 'image_url',
                        'image_url' => [
                            'url' => "data:{$mime_type};base64,{$base64_image}",
                            'detail' => 'low' // Use low detail to reduce payload size
                        ]
                    ];
                    
                    $content[] = $image_url_data;
                    $images_added++;
                    
                    error_log("Added small image {$filename} to multimodal analysis ({$file_size} bytes)");
                    
                } catch (Exception $e) {
                    error_log("Error processing image {$filename}: " . $e->getMessage());
                    continue;
                }
            }
        }
        
        error_log("Prepared simplified multimodal content: " . count($content) . " items ({$images_added} small images)");
        
        return $content;
    }

    /**
     * Build structured analysis prompt for modern OpenAI API
     * 
     * @param  array $lead_data Lead data
     * @param  array $parsed_files Parsed file contents
     * @return string Structured prompt
     */
    private function build_structured_analysis_prompt($lead_data, $parsed_files)
    {
        // Get the prompt template from options, fallback to default if not set
        $prompt_template = get_option('ai_analysis_prompt');
        if (empty($prompt_template)) {
            // Default enhanced prompt template
            $prompt_template = "You are an expert lead evaluation specialist. Analyze the provided lead information, follow-up notes, conversations, and any attached files to assess purchase likelihood.

=== LEAD INFORMATION ===
- Name: {lead_name}
- Email: {lead_email}
- Phone: {lead_phone}
- Company: {lead_company}
- Source: {lead_source}
- Value: {lead_value}
- Description: {lead_description}
- Date Added: {lead_date_added}
- Last Contact: {lead_last_contact}

=== FOLLOW-UP NOTES & CONVERSATIONS ===
{follow_up_notes}

=== ACTIVITY LOG ===
{activity_log}

=== ATTACHED FILES CONTENT ===
{file_content}

=== ANALYSIS INSTRUCTIONS ===
Analyze the lead information, follow-up notes, conversations, activity log, and any provided images/documents. Consider:
- Lead qualification and interest level based on conversations
- Communication patterns, response times, and engagement quality
- Follow-up types and frequency indicating interest level
- Staff notes about client interactions and feedback
- Budget indicators and financial capability from conversations
- Timeline and urgency signals from communications
- Objections, concerns, or positive signals mentioned in notes
- Social media presence and lifestyle (from images)
- Document content relevance
- Overall purchase intent signals from all touchpoints

IMPORTANT: Respond with valid JSON in this exact format:
{
  \"confidence_score\": 85,
  \"verdict\": \"Hot\",
  \"key_insights\": \"Analysis of behavior and communication patterns\",
  \"recommended_actions\": \"Specific next steps based on conversation history\",
  \"communication_analysis\": \"Assessment of engagement quality\",
  \"urgency_level\": \"High\",
  \"risk_factors\": [\"potential concerns from conversations\"],
  \"positive_signals\": [\"encouraging signs from interactions\"]
}

Ensure your response is valid JSON only, no additional text. Base your analysis heavily on the conversation history and follow-up notes.";
        }
        
        // Prepare follow-up notes section - sanitize content
        $follow_up_notes = '';
        if (isset($lead_data['notes']) && !empty($lead_data['notes'])) {
            $note_count = 0;
            foreach ($lead_data['notes'] as $note) {
                if ($note_count >= 10) break; // Limit to prevent payload issues
                
                $date = isset($note['dateadded']) ? date('Y-m-d H:i:s', strtotime($note['dateadded'])) : 'Unknown date';
                $staff_name = $this->sanitize_text($note['addedfrom_name'] ?? 'Staff');
                $description = $this->sanitize_text(strip_tags($note['description'] ?? ''));
                
                // Extract follow-up type if present
                if (preg_match('/Type:\s*([^<]+)/i', $note['description'] ?? '', $matches)) {
                    $followup_type = $this->sanitize_text(trim($matches[1]));
                    $description = $this->sanitize_text(strip_tags(preg_replace('/<strong>Type:[^<]*<\/strong><br>/', '', $note['description'] ?? '')));
                    
                    $follow_up_notes .= "Date: {$date} | Staff: {$staff_name} | Type: {$followup_type}\n";
                } else {
                    $follow_up_notes .= "Date: {$date} | Staff: {$staff_name}\n";
                }
                
                $follow_up_notes .= "Note: " . substr($description, 0, 500) . "\n"; // Limit note length
                
                if (!empty($note['date_contacted'])) {
                    $follow_up_notes .= "Contact Date: " . $this->sanitize_text($note['date_contacted']) . "\n";
                }
                
                $follow_up_notes .= "---\n";
                $note_count++;
            }
        }
        if (empty($follow_up_notes)) {
            $follow_up_notes = "No follow-up notes or conversations available.";
        }
        
        // Prepare activity log section - sanitize content
        $activity_log = '';
        if (isset($lead_data['activity_log']) && !empty($lead_data['activity_log'])) {
            $activity_count = 0;
            foreach ($lead_data['activity_log'] as $activity) {
                if ($activity_count >= 5) break; // Limit to recent 5 activities
                
                $date = isset($activity['date']) ? date('Y-m-d H:i:s', strtotime($activity['date'])) : 'Unknown date';
                $staff_name = $this->sanitize_text($activity['full_name'] ?? 'System');
                $description = $this->sanitize_text(strip_tags($activity['description'] ?? ''));
                
                $activity_log .= "Date: {$date} | {$staff_name}: " . substr($description, 0, 200) . "\n"; // Limit activity description
                $activity_count++;
            }
        }
        if (empty($activity_log)) {
            $activity_log = "No recent activity log available.";
        }
        
        // Prepare file content section - sanitize content
        $file_content = '';
        if (!empty($parsed_files)) {
            foreach ($parsed_files as $filename => $file_data) {
                if ($file_data['success'] && !empty($file_data['content'])) {
                    $safe_filename = $this->sanitize_text($filename);
                    $safe_content = $this->sanitize_text(substr($file_data['content'], 0, 1000)); // Limit file content
                    $file_content .= "File: {$safe_filename}\n";
                    $file_content .= "Content: {$safe_content}\n\n";
                }
            }
        }
        if (empty($file_content)) {
            $file_content = "No attached files available.";
        }
        
        // Replace placeholders in the template with actual data
        $replacements = [
            '{lead_name}' => $this->sanitize_text($lead_data['name'] ?? 'Not provided'),
            '{lead_email}' => $this->sanitize_text($lead_data['email'] ?? 'Not provided'),
            '{lead_phone}' => $this->sanitize_text($lead_data['phonenumber'] ?? 'Not provided'),
            '{lead_company}' => $this->sanitize_text($lead_data['company'] ?? 'Not provided'),
            '{lead_source}' => $this->sanitize_text($lead_data['source'] ?? 'Not provided'),
            '{lead_value}' => $this->sanitize_text($lead_data['lead_value'] ?? 'Not provided'),
            '{lead_description}' => $this->sanitize_text($lead_data['description'] ?? 'Not provided'),
            '{lead_date_added}' => $this->sanitize_text($lead_data['dateadded'] ?? 'Not provided'),
            '{lead_last_contact}' => $this->sanitize_text($lead_data['last_contact'] ?? 'Not provided'),
            '{follow_up_notes}' => $follow_up_notes,
            '{activity_log}' => $activity_log,
            '{file_content}' => $file_content
        ];
        
        // Apply the replacements to the template
        $prompt = str_replace(array_keys($replacements), array_values($replacements), $prompt_template);
        
        return $prompt;
    }
    
    /**
     * Build a simplified analysis prompt for fallback when regular prompt fails
     * 
     * @param array $lead_data Lead data
     * @return string Simplified prompt
     */
    private function build_simplified_prompt($lead_data)
    {
        $prompt = "Analyze this lead and provide a JSON response:\n\n";
        
        // Basic lead info only - no complex data that might cause encoding issues
        $prompt .= "Lead Name: " . $this->sanitize_text($lead_data['name'] ?? 'Unknown') . "\n";
        $prompt .= "Email: " . $this->sanitize_text($lead_data['email'] ?? 'Not provided') . "\n";
        $prompt .= "Phone: " . $this->sanitize_text($lead_data['phonenumber'] ?? 'Not provided') . "\n";
        
        if (!empty($lead_data['company'])) {
            $prompt .= "Company: " . $this->sanitize_text($lead_data['company']) . "\n";
        }
        
        if (!empty($lead_data['lead_value'])) {
            $prompt .= "Budget: " . $this->sanitize_text($lead_data['lead_value']) . "\n";
        }
        
        // Simple description without complex formatting
        if (!empty($lead_data['description'])) {
            $clean_desc = $this->sanitize_text(strip_tags($lead_data['description']));
            $prompt .= "Description: " . substr($clean_desc, 0, 200) . "\n";
        }
        
        // Count of interactions without details
        $note_count = isset($lead_data['notes']) ? count($lead_data['notes']) : 0;
        $activity_count = isset($lead_data['activity_log']) ? count($lead_data['activity_log']) : 0;
        
        $prompt .= "Follow-up notes: {$note_count}\n";
        $prompt .= "Activities: {$activity_count}\n\n";
        
        $prompt .= "Respond with valid JSON only:\n";
        $prompt .= "{\n";
        $prompt .= '  "confidence_score": 50,' . "\n";
        $prompt .= '  "verdict": "Warm",' . "\n";
        $prompt .= '  "key_insights": "Basic lead analysis completed",' . "\n";
        $prompt .= '  "recommended_actions": "Follow up with lead to gather more information",' . "\n";
        $prompt .= '  "communication_analysis": "Limited data available",' . "\n";
        $prompt .= '  "urgency_level": "Medium",' . "\n";
        $prompt .= '  "risk_factors": ["Limited information available"],' . "\n";
        $prompt .= '  "positive_signals": ["Lead provided contact information"]' . "\n";
        $prompt .= "}\n";
        
        return $prompt;
    }

    /**
     * Sanitize text content to ensure valid JSON encoding
     * 
     * @param string $text Text to sanitize
     * @return string Sanitized text
     */
    private function sanitize_text($text)
    {
        if (empty($text)) {
            return '';
        }
        
        // Convert to string if not already
        $text = (string)$text;
        
        // Remove or replace problematic characters
        $text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $text); // Remove control characters
        $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8'); // Ensure UTF-8 encoding
        $text = trim($text); // Remove leading/trailing whitespace
        
        // Replace problematic quotes and characters
        $text = str_replace(['"', "'", '`', "\r\n", "\r", "\n"], ['\\"', "\\'", "'", " ", " ", " "], $text);
        
        return $text;
    }

    /**
     * Make OpenAI API call
     * 
     * @param  string $prompt      AI prompt
     * @param  string $model       AI model
     * @param  float  $temperature Temperature setting
     * @param  int    $max_tokens  Max tokens
     * @return array              API response
     */
    private function make_openai_api_call($prompt, $model, $temperature, $max_tokens)
    {
        $api_key = get_option('openai_api_key');
        
        if (empty($api_key)) {
            return [
                'success' => false,
                'error' => 'OpenAI API key not configured'
            ];
        }
        
        $data = [
            'model' => $model,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => (float)$temperature,
            'max_tokens' => (int)$max_tokens
        ];
        
        // If prompt contains JSON format request, add response format
        if (strpos($prompt, 'JSON') !== false && strpos($prompt, 'response_format') === false) {
            $data['response_format'] = ['type' => 'json_object'];
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60); // Increased timeout for file processing
        
        $start_time = microtime(true);
        $response = curl_exec($ch);
        $response_time = microtime(true) - $start_time;
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        curl_close($ch);
        
        if ($response === false) {
            return [
                'success' => false,
                'error' => 'Failed to connect to OpenAI API'
            ];
        }
        
        $response_data = json_decode($response, true);
        
        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? 
                     $response_data['error']['message'] : 'Unknown API error';
            return [
                'success' => false,
                'error' => $error
            ];
        }
        
        if (!isset($response_data['choices'][0]['message']['content'])) {
            return [
                'success' => false,
                'error' => 'Invalid response from OpenAI API'
            ];
        }
        
        return [
            'success' => true,
            'response' => $response_data['choices'][0]['message']['content'],
            'response_time' => $response_time
        ];
    }

    /**
     * Make enhanced OpenAI API call with multimodal content support
     * 
     * @param  array  $content_array Array of content items (text, images)
     * @param  string $model Model to use
     * @param  float  $temperature Temperature setting
     * @param  int    $max_tokens Max tokens
     * @return array  API response
     */
    private function make_enhanced_openai_api_call($content_array, $model = 'gpt-4o', $temperature = 0.3, $max_tokens = 1000)
    {
        $api_key = get_option('openai_api_key');
        
        if (empty($api_key)) {
            return [
                'success' => false,
                'error' => 'OpenAI API key not configured'
            ];
        }
        
        $data = [
            'model' => $model,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $content_array
                ]
            ],
            'temperature' => (float)$temperature,
            'max_tokens' => (int)$max_tokens,
            'response_format' => [
                'type' => 'json_object'
            ]
        ];
        
        // Debug and validate the payload before sending
        $debug_result = $this->debug_json_payload($data);
        if (!$debug_result['valid']) {
            error_log('OpenAI API payload validation failed: ' . implode(', ', $debug_result['errors']));
            
            // If multimodal content fails, try text-only fallback
            if (is_array($content_array) && count($content_array) > 1) {
                error_log('Attempting text-only fallback due to multimodal validation failure');
                
                // Extract just the text content for fallback
                $text_content = '';
                foreach ($content_array as $item) {
                    if (isset($item['type']) && $item['type'] === 'text') {
                        $text_content = $item['text'];
                        break;
                    }
                }
                
                if (!empty($text_content)) {
                    // Retry with simple text-only API call
                    return $this->make_openai_api_call($text_content, $model, $temperature, $max_tokens);
                }
            }
            
            return [
                'success' => false,
                'error' => 'Request validation failed: ' . implode(', ', $debug_result['errors'])
            ];
        }
        
        // Log warnings if any
        if (!empty($debug_result['warnings'])) {
            error_log('OpenAI API payload warnings: ' . implode(', ', $debug_result['warnings']));
        }
        
        // Validate JSON encoding before sending
        $json_payload = json_encode($data, JSON_UNESCAPED_SLASHES);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'error' => 'JSON encoding error: ' . json_last_error_msg()
            ];
        }
        
        // Log the request size for debugging
        $payload_size = strlen($json_payload);
        error_log("OpenAI API request size: {$payload_size} bytes");
        
        // Check if payload is too large (OpenAI has limits)
        if ($payload_size > 32 * 1024 * 1024) { // 32MB limit
            return [
                'success' => false,
                'error' => 'Request payload too large: ' . round($payload_size / (1024 * 1024), 2) . 'MB'
            ];
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json_payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120); // Increased timeout for image processing
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // In case of SSL issues
        curl_setopt($ch, CURLOPT_VERBOSE, false); // Set to true for debugging
        
        $start_time = microtime(true);
        $response = curl_exec($ch);
        $response_time = microtime(true) - $start_time;
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        
        curl_close($ch);
        
        if ($response === false) {
            return [
                'success' => false,
                'error' => 'Failed to connect to OpenAI API: ' . $curl_error
            ];
        }
        
        $response_data = json_decode($response, true);
        
        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? 
                     $response_data['error']['message'] : 'Unknown API error';
            
            // Log the actual request for debugging if there's an error
            if ($http_code >= 400) {
                error_log("OpenAI API Error {$http_code}: {$error}");
                error_log("Request payload preview: " . substr($json_payload, 0, 1000) . "...");
            }
            
            return [
                'success' => false,
                'error' => $error,
                'http_code' => $http_code
            ];
        }
        
        if (!isset($response_data['choices'][0]['message']['content'])) {
            return [
                'success' => false,
                'error' => 'Invalid response from OpenAI API'
            ];
        }
        
        return [
            'success' => true,
            'response' => $response_data['choices'][0]['message']['content'],
            'response_time' => $response_time,
            'usage' => $response_data['usage'] ?? null
        ];
    }
    
    /**
     * Parse structured AI response (expects JSON)
     * 
     * @param  string $response Raw AI response
     * @return array  Parsed analysis data
     */
    private function parse_structured_ai_response($response)
    {
        // Try to parse as JSON first
        $json_data = json_decode(trim($response), true);
        
        if (json_last_error() === JSON_ERROR_NONE && is_array($json_data)) {
            // Structured JSON response - handle both old and new format
            $score = (int)($json_data['confidence_score'] ?? $json_data['score'] ?? 0);
            $verdict = $json_data['verdict'] ?? 'Cold';
            
            // Validate verdict values
            if (!in_array($verdict, ['Hot', 'Warm', 'Cold', 'Qualified', 'Likely', 'Unlikely'])) {
                $verdict = 'Cold';
            }
            
            return [
                'score' => min(100, max(0, $score)),
                'verdict' => $verdict,
                'rationale' => $json_data['key_insights'] ?? $json_data['rationale'] ?? 'Analysis completed',
                'recommendations' => $json_data['recommended_actions'] ?? $json_data['recommendations'] ?? '',
                'key_insights' => is_array($json_data['key_insights'] ?? null) ? 
                                 $json_data['key_insights'] : 
                                 [$json_data['key_insights'] ?? 'No specific insights'],
                'risk_factors' => $json_data['risk_factors'] ?? [],
                'confidence_level' => $json_data['urgency_level'] ?? $json_data['confidence_level'] ?? 'Medium',
                'communication_analysis' => $json_data['communication_analysis'] ?? '',
                'positive_signals' => $json_data['positive_signals'] ?? []
            ];
        } else {
            // Fallback: try to extract basic info from text response
            error_log('AI response was not valid JSON, falling back to text parsing');
            
            $score = 50; // Default score
            $verdict = 'Cold'; // Default verdict
            $rationale = trim($response);
            
            // Try to extract score from text
            if (preg_match('/score[:\s]*(\d+)/i', $response, $matches)) {
                $score = (int)$matches[1];
            }
            
            // Try to extract verdict from text
            if (preg_match('/\b(hot|warm|qualified)\b/i', $response, $matches)) {
                $verdict = ucfirst(strtolower($matches[1]));
            } elseif (preg_match('/likely/i', $response)) {
                $verdict = 'Warm';
            }
            
            return [
                'score' => min(100, max(0, $score)),
                'verdict' => $verdict,
                'rationale' => $rationale,
                'recommendations' => 'Please review the raw AI response for detailed recommendations.',
                'key_insights' => ['Raw AI response available'],
                'risk_factors' => [],
                'confidence_level' => 'Low',
                'communication_analysis' => '',
                'positive_signals' => []
            ];
        }
    }
    
    /**
     * Debug and validate JSON payload before sending to OpenAI
     * 
     * @param  array $data Request data to validate
     * @return array Validation result
     */
    private function debug_json_payload($data)
    {
        $debug_info = [
            'valid' => false,
            'size' => 0,
            'errors' => [],
            'warnings' => []
        ];
        
        try {
            // Test JSON encoding
            $json = json_encode($data, JSON_UNESCAPED_SLASHES);
            $debug_info['size'] = strlen($json);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                $debug_info['errors'][] = 'JSON encoding error: ' . json_last_error_msg();
                return $debug_info;
            }
            
            // Check payload size
            $size_mb = $debug_info['size'] / (1024 * 1024);
            if ($size_mb > 25) {
                $debug_info['warnings'][] = "Large payload: {$size_mb}MB - may cause issues";
            }
            
            // Validate structure
            if (!isset($data['model'])) {
                $debug_info['errors'][] = 'Missing model parameter';
            }
            
            if (!isset($data['messages']) || !is_array($data['messages'])) {
                $debug_info['errors'][] = 'Invalid messages structure';
            }
            
            // Check content array structure
            if (isset($data['messages'][0]['content']) && is_array($data['messages'][0]['content'])) {
                $content_count = count($data['messages'][0]['content']);
                $image_count = 0;
                
                foreach ($data['messages'][0]['content'] as $item) {
                    if (isset($item['type']) && $item['type'] === 'image_url') {
                        $image_count++;
                        
                        // Check image URL format
                        if (!isset($item['image_url']['url'])) {
                            $debug_info['errors'][] = 'Missing image URL in content item';
                        } elseif (!preg_match('/^data:image\/[^;]+;base64,/', $item['image_url']['url'])) {
                            $debug_info['errors'][] = 'Invalid image URL format';
                        }
                    }
                }
                
                $debug_info['content_items'] = $content_count;
                $debug_info['image_items'] = $image_count;
                
                if ($image_count > 10) {
                    $debug_info['warnings'][] = "Many images ({$image_count}) - may exceed API limits";
                }
            }
            
            $debug_info['valid'] = empty($debug_info['errors']);
            
        } catch (Exception $e) {
            $debug_info['errors'][] = 'Exception during validation: ' . $e->getMessage();
        }
        
        return $debug_info;
    }
}
