<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * CRM Event Logger Class
 * Provides simple and advanced logging capabilities for critical CRM actions
 */
class Crm_logger
{
    private $CI;
    private $log_path;

    public function __construct()
    {
        $this->CI = &get_instance();
        $this->log_path = APPPATH . 'logs/';
        
        // Make sure log directory exists
        if (!is_dir($this->log_path)) {
            mkdir($this->log_path, 0755, true);
        }
    }
    
    /**
     * Write to the error log
     * 
     * @param string $message Message to log
     * @param string $level Log level (debug, info, warning, error, critical)
     * @param array $context Additional context data
     */
    public function write_log($message, $level = 'info', $context = [])
    {
        $timestamp = date('Y-m-d H:i:s');
        $log_file = $this->log_path . 'crm_log-' . date('Y-m-d') . '.php';
        
        // Create file with PHP header if it doesn't exist
        if (!file_exists($log_file)) {
            file_put_contents($log_file, "<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>\n");
        }
        
        // Get user info
        $user_id = 'Guest';
        $username = 'Guest';
        
        if ($this->CI->session->has_userdata('staff_user_id')) {
            $user_id = $this->CI->session->userdata('staff_user_id');
            $username = get_staff_full_name($user_id) ?: 'Unknown';
        }
        
        // Format context as JSON if not empty
        $context_str = '';
        if (!empty($context)) {
            $context_str = json_encode($context);
        }
        
        // Format the log entry
        $entry = sprintf(
            "[%s] [%s] [User:%s:%s] [%s] %s %s\n",
            $timestamp,
            strtoupper($level),
            $user_id,
            $username,
            $this->CI->input->ip_address(),
            $message,
            $context_str ? "Context: " . $context_str : ""
        );
        
        // Write to file
        file_put_contents($log_file, $entry, FILE_APPEND);
        
        // If critical or error, also write to PHP error log
        if (in_array(strtolower($level), ['critical', 'error'])) {
            error_log($entry);
        }
        
        return true;
    }
    
    /**
     * Log a database related event
     * 
     * @param string $table Table name
     * @param string $action Action performed (insert, update, delete)
     * @param mixed $record_id ID of the affected record
     * @param array $data Additional data
     */
    public function log_db_action($table, $action, $record_id, $data = [])
    {
        $message = sprintf("Database %s on table %s for record #%s", $action, $table, $record_id);
        $this->write_log($message, 'info', $data);
    }
    
    /**
     * Log a user action
     * 
     * @param string $action Action performed
     * @param string $section Section of the CRM (leads, clients, invoices, etc.)
     * @param mixed $item_id ID of the affected item
     * @param array $data Additional data
     */
    public function log_user_action($action, $section, $item_id = null, $data = [])
    {
        $message = sprintf("User action: %s in %s", $action, $section);
        if ($item_id) {
            $message .= " #" . $item_id;
        }
        $this->write_log($message, 'info', $data);
    }
    
    /**
     * Log an error
     * 
     * @param string $message Error message
     * @param string $location Where the error occurred
     * @param array $data Additional data
     */
    public function log_error($message, $location, $data = [])
    {
        $message = sprintf("[%s] %s", $location, $message);
        $this->write_log($message, 'error', $data);
    }
}
