<?php

use app\services\imap\Imap;
use app\services\LeadProfileBadges;
use app\services\leads\LeadsKanban;
use app\services\imap\ConnectionErrorException;
use Ddeboer\Imap\Exception\MailboxDoesNotExistException;

header('Content-Type: text/html; charset=utf-8');
defined('BASEPATH') or exit('No direct script access allowed');

// This code has been moved to manage follow-up types in the leads settings

class Leads extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('leads_model');
        
        // Handle CSRF for AI analysis methods - ensure they're properly protected
        $ai_methods = ['ai_analyze', 'ai_upload_files', 'get_ai_report'];
        if (in_array($this->router->method, $ai_methods)) {
            // Load security library for CSRF handling
            $this->load->library('security');
            
            // For GET requests (like get_ai_report), we don't need CSRF
            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                // CSRF not needed for GET requests
            } else {
                // For POST requests, ensure CSRF is properly handled
                // CodeIgniter will automatically validate CSRF if enabled
            }
        }
    }
    
    /**
     * Get lead data via AJAX for reminder notifications
     */
    public function get_lead_data_ajax()
    {
        // Set headers for AJAX
        header('Content-Type: application/json');
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: POST');
        header('Access-Control-Allow-Headers: Content-Type');
        
        // Check if user is logged in
        if (!is_staff_logged_in()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }
        
        // Get lead ID from POST
        $lead_id = $this->input->post('lead_id');
        
        if (!$lead_id) {
            echo json_encode(['success' => false, 'message' => 'Lead ID is required']);
            return;
        }
        
        // Get lead data
        $lead = $this->leads_model->get($lead_id);
        
        if (!$lead) {
            echo json_encode(['success' => false, 'message' => 'Lead not found']);
            return;
        }
        
        // Format lead data for response
        $lead_data = [
            'success' => true,
            'lead' => [
                'id' => $lead->id,
                'name' => $lead->name,
                'email' => $lead->email,
                'phonenumber' => $lead->phonenumber,
                'company' => $lead->company,
                'status_name' => $lead->status_name,
                'source_name' => $lead->source_name
            ]
        ];
        
        // Return lead data as JSON
        echo json_encode($lead_data);
    }

    /* List all leads */
    public function index($id = '')
    {
        close_setup_menu();

        if (!is_staff_member()) {
            access_denied('Leads');
        }

        $data['switch_kanban'] = true;

        if ($this->session->userdata('leads_kanban_view') == 'true') {
            $data['switch_kanban'] = false;
            $data['bodyclass']     = 'kan-ban-body';
        }

        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
            $this->load->model('gdpr_model');
            $data['consent_purposes'] = $this->gdpr_model->get_consent_purposes();
        }
        $data['summary']  = get_leads_summary();
        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();
        $data['title']    = _l('leads');
        $data['table'] = App_table::find('leads');
        // in case accesed the url leads/index/ directly with id - used in search
        $data['leadid']   = $id;
        $data['isKanBan'] = $this->session->has_userdata('leads_kanban_view') &&
            $this->session->userdata('leads_kanban_view') == 'true';

        $this->load->view('admin/leads/manage_leads', $data);
    }

    public function table()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        App_table::find('leads')->output();
    }

    public function kanban()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $data['statuses']      = $this->leads_model->get_status();
        $data['base_currency'] = get_base_currency();
        $data['summary']       = get_leads_summary();

        echo $this->load->view('admin/leads/kan-ban', $data, true);
    }

    /* Add or update lead */
    public function lead($id = '')
    {
        if (!is_staff_member() || ($id != '' && !$this->leads_model->staff_can_access_lead($id))) {
            ajax_access_denied();
        }

        if ($this->input->post()) {
            $data = $this->input->post();
            $message = '';
            $proposalWarning = false;
            
                $phoneWarning = false;
            
                // Check for spaces in phone numbers
                if (!empty($data['phonenumber']) && strpos($data['phonenumber'], ' ') !== false) {
                    $phoneWarning = true;
                }
            
                // Check additional phone numbers for spaces
                if (!empty($data['additional_phones']) && is_array($data['additional_phones'])) {
                    foreach ($data['additional_phones'] as $phone) {
                        if (!empty($phone) && strpos($phone, ' ') !== false) {
                            $phoneWarning = true;
                            break;
                        }
                    }
                }
            
                // Log the incoming address fields for debugging
            app_log("Lead form submission - Address Fields", 'debug', [
                'lead_id' => $id,
                'address' => $data['address'] ?? null,
                'city' => $data['city'] ?? null,
                'state' => $data['state'] ?? null,
                'country' => $data['country'] ?? null,
                'zip' => $data['zip'] ?? null
            ]);

            if ($id == '') {
                $id = $this->leads_model->add($data);
                $message = $id ? _l('added_successfully', _l('lead')) : '';

                echo json_encode([
                    'success'      => $id ? true : false,
                    'id'           => $id,
                    'message'      => $message,
                    'leadView'     => $id ? $this->_get_lead_data($id) : [],
                    'phoneWarning' => $phoneWarning,
                ]);
            } else {
                $emailOriginal = $this->db->select('email')->where('id', $id)->get(db_prefix() . 'leads')->row()->email;
                $success = $this->leads_model->update($data, $id);

                if ($success) {
                    $emailNow = $this->db->select('email')->where('id', $id)->get(db_prefix() . 'leads')->row()->email;

                    $proposalWarning = (total_rows(db_prefix() . 'proposals', [
                        'rel_type' => 'lead',
                        'rel_id'   => $id, ]) > 0 && ($emailOriginal != $emailNow) && $emailNow != '') ? true : false;

                    $message = _l('updated_successfully', _l('lead'));

                    // Log the update result
                    app_log("Lead updated successfully with address fields", 'info', [
                        'lead_id' => $id,
                        'address_fields_updated' => [
                            'address' => $data['address'] ?? null,
                            'city' => $data['city'] ?? null,
                            'state' => $data['state'] ?? null,
                            'country' => $data['country'] ?? null,
                            'zip' => $data['zip'] ?? null
                        ]
                    ]);
                }

                echo json_encode([
                    'success'          => $success,
                    'message'          => $message,
                    'id'              => $id,
                    'proposal_warning' => $proposalWarning,
                        'phoneWarning'     => $phoneWarning,
                    'leadView'        => $this->_get_lead_data($id),
                ]);
            }
            die;
        }

        echo json_encode([
            'leadView' => $this->_get_lead_data($id),
        ]);
    }

    private function _get_lead_data($id = '')
    {
        $reminder_data         = '';
        $data['lead_locked']   = false;
        $data['openEdit']      = $this->input->get('edit') ? true : false;
        $data['members']       = $this->staff_model->get('', ['is_not_staff' => 0, 'active' => 1]);
        $data['status_id']     = $this->input->get('status_id') ? $this->input->get('status_id') : get_option('leads_default_status');
        $data['base_currency'] = get_base_currency();

        if (is_numeric($id)) {
            $leadWhere = (staff_can('view',  'leads') ? [] : '(assigned = ' . get_staff_user_id() . ' OR addedfrom=' . get_staff_user_id() . ' OR is_public=1)');

            $lead = $this->leads_model->get($id, $leadWhere);

            if (!$lead) {
                header('HTTP/1.0 404 Not Found');
                echo _l('lead_not_found');
                die;
            }

            if (total_rows(db_prefix() . 'clients', ['leadid' => $id ]) > 0) {
                $data['lead_locked'] = ((!is_admin() && get_option('lead_lock_after_convert_to_customer') == 1) ? true : false);
            }

            $reminder_data = $this->load->view('admin/includes/modals/reminder', [
                    'id'             => $lead->id,
                    'name'           => 'lead',
                    'members'        => $data['members'],
                    'reminder_title' => _l('lead_set_reminder_title'),
                ], true);

            $data['lead']          = $lead;
            $data['mail_activity'] = $this->leads_model->get_mail_activity($id);
            $data['notes']         = $this->leads_model->get_lead_notes_with_followup('', $id);
            $data['activity_log']  = $this->leads_model->get_lead_activity_log($id);
            
            // Get latest dealer interest status
            $this->db->select('dealer_interest');
            $this->db->where('leadid', $id);
            $this->db->where('dealer_interest IS NOT NULL');
            $this->db->where('dealer_interest !=', '');
            $this->db->order_by('dateadded', 'DESC');
            $this->db->limit(1);
            $latest_dealer_interest = $this->db->get(db_prefix() . 'lead_notes')->row();
            $data['latest_dealer_interest'] = $latest_dealer_interest ? $latest_dealer_interest->dealer_interest : '';

            if (is_gdpr() && get_option('gdpr_enable_consent_for_leads') == '1') {
                $this->load->model('gdpr_model');
                $data['purposes'] = $this->gdpr_model->get_consent_purposes($lead->id, 'lead');
                $data['consents'] = $this->gdpr_model->get_consents(['lead_id' => $lead->id]);
            }

            $leadProfileBadges         = new LeadProfileBadges($id);
            $data['total_reminders']   = $leadProfileBadges->getCount('reminders');
            $data['total_notes']       = $leadProfileBadges->getCount('notes');
            $data['total_attachments'] = $leadProfileBadges->getCount('attachments');
            $data['total_tasks']       = $leadProfileBadges->getCount('tasks');
            $data['total_proposals']   = $leadProfileBadges->getCount('proposals');
        }


        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();
        
        // Add follow-up types to the view data
        $followup_types = get_option('lead_followup_types');
        $data['followup_types'] = $followup_types ? json_decode($followup_types, true) : ['Call', 'Message'];
        if (!is_array($data['followup_types'])) {
            $data['followup_types'] = ['Call', 'Message'];
        }

        $data = hooks()->apply_filters('lead_view_data', $data);

        return [
            'data'          => $this->load->view('admin/leads/lead', $data, true),
            'reminder_data' => $reminder_data,
        ];
    }

    public function leads_kanban_load_more()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $status = $this->input->get('status');
        $page   = $this->input->get('page');

        $this->db->where('id', $status);
        $status = $this->db->get(db_prefix() . 'leads_status')->row_array();

        $leads = (new LeadsKanban($status['id']))
        ->search($this->input->get('search'))
        ->sortBy(
            $this->input->get('sort_by'),
            $this->input->get('sort')
        )
        ->page($page)->get();

        foreach ($leads as $lead) {
            $this->load->view('admin/leads/_kan_ban_card', [
                'lead'   => $lead,
                'status' => $status,
            ]);
        }
    }

    public function switch_kanban($set = 0)
    {
        if ($set == 1) {
            $set = 'true';
        } else {
            $set = 'false';
        }
        $this->session->set_userdata([
            'leads_kanban_view' => $set,
        ]);
        redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
    }

    public function export($id)
    {
        if (is_admin()) {
            $this->load->library('gdpr/gdpr_lead');
            $this->gdpr_lead->export($id);
        }
    }

    /* Delete lead from database */
    public function delete($id)
    {
        if (!$id) {
            redirect(admin_url('leads'));
        }

        if (staff_cant('delete', 'leads')) {
            access_denied('Delete Lead');
        }

        $response = $this->leads_model->delete($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_lowercase')));
        } elseif ($response === true) {
            set_alert('success', _l('deleted', _l('lead')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_lowercase')));
        }

        $ref = $_SERVER['HTTP_REFERER'];

        // if user access leads/inded/ID to prevent redirecting on the same url because will throw 404
        if (!$ref || strpos($ref, 'index/' . $id) !== false) {
            redirect(admin_url('leads'));
        }

        redirect($ref);
    }

    public function mark_as_lost($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->mark_as_lost($id);
        if ($success) {
            $message = _l('lead_marked_as_lost');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function unmark_as_lost($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->unmark_as_lost($id);
        if ($success) {
            $message = _l('lead_unmarked_as_lost');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function mark_as_junk($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->mark_as_junk($id);
        if ($success) {
            $message = _l('lead_marked_as_junk');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function unmark_as_junk($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        $message = '';
        $success = $this->leads_model->unmark_as_junk($id);
        if ($success) {
            $message = _l('lead_unmarked_as_junk');
        }
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($id),
            'id'       => $id,
        ]);
    }

    public function add_activity()
    {
        $leadid = $this->input->post('leadid');
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($leadid)) {
            ajax_access_denied();
        }
        if ($this->input->post()) {
            $message = $this->input->post('activity');
            $aId     = $this->leads_model->log_lead_activity($leadid, $message);
            if ($aId) {
                $this->db->where('id', $aId);
                $this->db->update(db_prefix() . 'lead_activity_log', ['custom_activity' => 1]);
            }
            echo json_encode(['leadView' => $this->_get_lead_data($leadid), 'id' => $leadid]);
        }
    }

    public function get_convert_data($id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }
        if (is_gdpr() && get_option('gdpr_enable_consent_for_contacts') == '1') {
            $this->load->model('gdpr_model');
            $data['purposes'] = $this->gdpr_model->get_consent_purposes($id, 'lead');
        }
        $data['lead'] = $this->leads_model->get($id);
        $this->load->view('admin/leads/convert_to_customer', $data);
    }

    /**
     * Convert lead to client
     * @since  version 1.0.1
     * @return mixed
     */
    public function convert_to_customer()
    {
        if (!is_staff_member()) {
            access_denied('Lead Convert to Customer');
        }

        if ($this->input->post()) {
            $default_country  = get_option('customer_default_country');
            $data             = $this->input->post();
            $data['password'] = $this->input->post('password', false);

            $original_lead_email = $data['original_lead_email'];
            unset($data['original_lead_email']);

            if (isset($data['transfer_notes'])) {
                $notes = $this->misc_model->get_notes($data['leadid'], 'lead');
                unset($data['transfer_notes']);
            }

            if (isset($data['transfer_consent'])) {
                $this->load->model('gdpr_model');
                $consents = $this->gdpr_model->get_consents(['lead_id' => $data['leadid']]);
                unset($data['transfer_consent']);
            }

            if (isset($data['merge_db_fields'])) {
                $merge_db_fields = $data['merge_db_fields'];
                unset($data['merge_db_fields']);
            }

            if (isset($data['merge_db_contact_fields'])) {
                $merge_db_contact_fields = $data['merge_db_contact_fields'];
                unset($data['merge_db_contact_fields']);
            }

            if (isset($data['include_leads_custom_fields'])) {
                $include_leads_custom_fields = $data['include_leads_custom_fields'];
                unset($data['include_leads_custom_fields']);
            }

            if ($data['country'] == '' && $default_country != '') {
                $data['country'] = $default_country;
            }

            $data['billing_street'  ] = $data['address'];
            $data['billing_city'    ] = $data['city'];
            $data['billing_state'   ] = $data['state'];
            $data['billing_zip'     ] = $data['zip'];
            $data['billing_country' ] = $data['country'];

            $data['is_primary'] = 1;
            $id                 = $this->clients_model->add($data, true);
            if ($id) {
                $primary_contact_id = get_primary_contact_user_id($id);

                if (isset($notes)) {
                    foreach ($notes as $note) {
                        $this->db->insert(db_prefix() . 'notes', [
                            'rel_id'         => $id,
                            'rel_type'       => 'customer',
                            'dateadded'      => $note['dateadded'],
                            'addedfrom'      => $note['addedfrom'],
                            'description'    => $note['description'],
                            'date_contacted' => $note['date_contacted'],
                            ]);
                    }
                }
                if (isset($consents)) {
                    foreach ($consents as $consent) {
                        unset($consent['id']);
                        unset($consent['purpose_name']);
                        $consent['lead_id']    = 0;
                        $consent['contact_id'] = $primary_contact_id;
                        $this->gdpr_model->add_consent($consent);
                    }
                }
                if (staff_cant('view', 'customers') && get_option('auto_assign_customer_admin_after_lead_convert') == 1) {
                    $this->db->insert(db_prefix() . 'customer_admins', [
                        'date_assigned' => date('Y-m-d H:i:s'),
                        'customer_id'   => $id,
                        'staff_id'      => get_staff_user_id(),
                    ]);
                }
                $this->leads_model->log_lead_activity($data['leadid'], 'not_lead_activity_converted', false, serialize([
                    get_staff_full_name(),
                ]));
                $default_status = $this->leads_model->get_status('', [
                    'isdefault' => 1,
                ]);
                $this->db->where('id', $data['leadid']);
                $this->db->update(db_prefix() . 'leads', [
                    'date_converted' => date('Y-m-d H:i:s'),
                    'status'         => $default_status[0]['id'],
                    'junk'           => 0,
                    'lost'           => 0,
                ]);
                // Check if lead email is different then client email
                $contact = $this->clients_model->get_contact(get_primary_contact_user_id($id));
                if ($contact->email != $original_lead_email) {
                    if ($original_lead_email != '') {
                        $this->leads_model->log_lead_activity($data['leadid'], 'not_lead_activity_converted_email', false, serialize([
                            $original_lead_email,
                            $contact->email,
                        ]));
                    }
                }
                if (isset($include_leads_custom_fields)) {
                    foreach ($include_leads_custom_fields as $fieldid => $value) {
                        // checked don't merge
                        if ($value == 5) {
                            continue;
                        }
                        // get the value of this leads custom fiel
                        $this->db->where('relid', $data['leadid']);
                        $this->db->where('fieldto', 'leads');
                        $this->db->where('fieldid', $fieldid);
                        $lead_custom_field_value = $this->db->get(db_prefix() . 'customfieldsvalues')->row()->value;
                        // Is custom field for contact ot customer
                        if ($value == 1 || $value == 4) {
                            if ($value == 4) {
                                $field_to = 'contacts';
                            } else {
                                $field_to = 'customers';
                            }
                            $this->db->where('id', $fieldid);
                            $field = $this->db->get(db_prefix() . 'customfields')->row();
                            // check if this field exists for custom fields
                            $this->db->where('fieldto', $field_to);
                            $this->db->where('name', $field->name);
                            $exists               = $this->db->get(db_prefix() . 'customfields')->row();
                            $copy_custom_field_id = null;
                            if ($exists) {
                                $copy_custom_field_id = $exists->id;
                            } else {
                                // there is no name with the same custom field for leads at the custom side create the custom field now
                                $this->db->insert(db_prefix() . 'customfields', [
                                    'fieldto'        => $field_to,
                                    'name'           => $field->name,
                                    'required'       => $field->required,
                                    'type'           => $field->type,
                                    'options'        => $field->options,
                                    'display_inline' => $field->display_inline,
                                    'field_order'    => $field->field_order,
                                    'slug'           => slug_it($field_to . '_' . $field->name, [
                                        'separator' => '_',
                                    ]),
                                    'active'        => $field->active,
                                    'only_admin'    => $field->only_admin,
                                    'show_on_table' => $field->show_on_table,
                                    'bs_column'     => $field->bs_column,
                                ]);
                                $new_customer_field_id = $this->db->insert_id();
                                if ($new_customer_field_id) {
                                    $copy_custom_field_id = $new_customer_field_id;
                                }
                            }
                            if ($copy_custom_field_id != null) {
                                $insert_to_custom_field_id = $id;
                                if ($value == 4) {
                                    $insert_to_custom_field_id = get_primary_contact_user_id($id);
                                }
                                $this->db->insert(db_prefix() . 'customfieldsvalues', [
                                    'relid'   => $insert_to_custom_field_id,
                                    'fieldid' => $copy_custom_field_id,
                                    'fieldto' => $field_to,
                                    'value'   => $lead_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 2) {
                            if (isset($merge_db_fields)) {
                                $db_field = $merge_db_fields[$fieldid];
                                // in case user don't select anything from the db fields
                                if ($db_field == '') {
                                    continue;
                                }
                                if ($db_field == 'country' || $db_field == 'shipping_country' || $db_field == 'billing_country') {
                                    $this->db->where('iso2', $lead_custom_field_value);
                                    $this->db->or_where('short_name', $lead_custom_field_value);
                                    $this->db->or_like('long_name', $lead_custom_field_value);
                                    $country = $this->db->get(db_prefix() . 'countries')->row();
                                    if ($country) {
                                        $lead_custom_field_value = $country->country_id;
                                    } else {
                                        $lead_custom_field_value = 0;
                                    }
                                }
                                $this->db->where('userid', $id);
                                $this->db->update(db_prefix() . 'clients', [
                                    $db_field => $lead_custom_field_value,
                                ]);
                            }
                        } elseif ($value == 3) {
                            if (isset($merge_db_contact_fields)) {
                                $db_field = $merge_db_contact_fields[$fieldid];
                                if ($db_field == '') {
                                    continue;
                                }
                                $this->db->where('id', $primary_contact_id);
                                $this->db->update(db_prefix() . 'contacts', [
                                    $db_field => $lead_custom_field_value,
                                ]);
                            }
                        }
                    }
                }
                // set the lead to status client in case is not status client
                $this->db->where('isdefault', 1);
                $status_client_id = $this->db->get(db_prefix() . 'leads_status')->row()->id;
                $this->db->where('id', $data['leadid']);
                $this->db->update(db_prefix() . 'leads', [
                    'status' => $status_client_id,
                ]);

                set_alert('success', _l('lead_to_client_base_converted_success'));

                if (is_gdpr() && get_option('gdpr_after_lead_converted_delete') == '1') {
                    // When lead is deleted
                    // move all proposals to the actual customer record
                    $this->db->where('rel_id', $data['leadid']);
                    $this->db->where('rel_type', 'lead');
                    $this->db->update('proposals', [
                        'rel_id'   => $id,
                        'rel_type' => 'customer',
                    ]);

                    $this->leads_model->delete($data['leadid']);

                    $this->db->where('userid', $id);
                    $this->db->update(db_prefix() . 'clients', ['leadid' => null]);
                }

                log_activity('Created Lead Client Profile [LeadID: ' . $data['leadid'] . ', ClientID: ' . $id . ']');
                hooks()->do_action('lead_converted_to_customer', ['lead_id' => $data['leadid'], 'customer_id' => $id]);
                redirect(admin_url('clients/client/' . $id));
            }
        }
    }

    /* Used in kanban when dragging and mark as */
    public function update_lead_status()
    {
        // Set JSON content type header
        header('Content-Type: application/json');
        
        if (!$this->input->post() || !$this->input->is_ajax_request()) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid request'
            ]);
            exit;
        }

        if (!is_staff_member()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            exit;
        }

        $post_data = $this->input->post();
        $lead_id = isset($post_data['leadid']) ? $post_data['leadid'] : null;

        if (!$lead_id) {
            echo json_encode([
                'success' => false,
                'message' => 'Lead ID is required'
            ]);
            exit;
        }

        if (!$this->leads_model->staff_can_access_lead($lead_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'You do not have permission to update this lead'
            ]);
            exit;
        }

        try {
            $success = $this->leads_model->update_lead_status($post_data);
            
            echo json_encode([
                'success' => $success,
                'message' => $success ? _l('updated_successfully', _l('lead_status')) : _l('something_went_wrong')
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error updating lead status: ' . $e->getMessage());
            echo json_encode([
                'success' => false,
                'message' => 'An error occurred while updating the lead status'
            ]);
        }
        exit;
    }

    public function update_status_order()
    {
        if ($post_data = $this->input->post()) {
            $this->leads_model->update_status_order($post_data);
        }
    }

    public function add_lead_attachment()
    {
        $id       = $this->input->post('id');
        $lastFile = $this->input->post('last_file');
        $description = $this->input->post('description');

        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($id)) {
            ajax_access_denied();
        }

        // Debug the received data with more details
        error_log('Lead attachment upload - POST data: ' . print_r($_POST, true));
        error_log('Lead attachment upload - ID: ' . $id . ', Description: ' . $description);
        
        // Ensure description is available for the upload handler
        $_POST['description'] = $description;
        
        // Store description in session to be used by handle_lead_attachments
        $this->session->set_userdata('lead_attachment_description', $description);
        
        // Store it in a global variable too as a fallback
        $GLOBALS['lead_attachment_description'] = $description;

        handle_lead_attachments($id);
        echo json_encode(['leadView' => $lastFile ? $this->_get_lead_data($id) : [], 'id' => $id]);
    }

    public function add_external_attachment()
    {
        if ($this->input->post()) {
            $this->leads_model->add_attachment_to_database(
                $this->input->post('lead_id'),
                $this->input->post('files'),
                $this->input->post('external')
            );
        }
    }

    public function delete_attachment($id, $lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }
        echo json_encode([
            'success'  => $this->leads_model->delete_lead_attachment($id),
            'leadView' => $this->_get_lead_data($lead_id),
            'id'       => $lead_id,
        ]);
    }

    public function delete_note($id, $lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }
        
        // Delete from lead_notes table instead of notes table
        $this->db->where('id', $id);
        $this->db->where('leadid', $lead_id);
        $success = $this->db->delete(db_prefix() . 'lead_notes');
        
        echo json_encode([
            'success'  => $success,
            'leadView' => $this->_get_lead_data($lead_id),
            'id'       => $lead_id,
        ]);
    }
    
    public function edit_note($id)
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }
        
        if ($this->input->post()) {
            $data = $this->input->post();
            
            // Check if user has permission to edit this note
            $this->db->where('id', $id);
            $note = $this->db->get(db_prefix() . 'lead_notes')->row();
            
            if (!$note || ($note->addedfrom != get_staff_user_id() && !is_admin())) {
                echo json_encode([
                    'success' => false,
                    'message' => 'You do not have permission to edit this note'
                ]);
                return;
            }
            
            // Update the note in lead_notes table
            $this->db->where('id', $id);
            $success = $this->db->update(db_prefix() . 'lead_notes', [
                'description' => nl2br($data['description'])
            ]);
            
            echo json_encode([
                'description' => process_text_content_for_display(nl2br($data['description'])),
                'success'     => $success,
                'message'     => $success ? _l('note_updated_successfully') : _l('something_went_wrong'),
            ]);
        }
    }
    
    /**
     * Add lead note
     * @param mixed $lead_id lead id
     */
    public function add_note($lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }
        
        if ($this->input->post()) {
            // LEVEL 1: Rate limiting - prevent more than 1 request per 2 seconds per user per lead
            $rate_limit_key = 'rate_limit_' . get_staff_user_id() . '_' . $lead_id;
            $last_request_time = $this->session->userdata($rate_limit_key);
            
            if ($last_request_time && (time() - $last_request_time) < 2) {
                echo json_encode([
                    'success'  => false,
                    'message'  => 'Please wait before submitting another follow-up.',
                    'rate_limited' => true
                ]);
                return;
            }
            
            $this->session->set_userdata($rate_limit_key, time());
            $this->session->mark_as_temp($rate_limit_key, 10); // 10 seconds
            
            // LEVEL 2: IMMEDIATE duplicate prevention - check if this exact request was already processed
            $request_signature = md5($lead_id . '_' . get_staff_user_id() . '_' . $this->input->post('description') . '_' . date('Y-m-d H:i'));
            $immediate_check_key = 'processed_' . $request_signature;
            
            if ($this->session->userdata($immediate_check_key)) {
                // This exact request was already processed within the last minute
                echo json_encode([
                    'success'  => true,
                    'message'  => _l('added_successfully', _l('followup')),
                    'leadView' => $this->_get_lead_data($lead_id),
                    'id'       => $lead_id,
                    'duplicate_prevented' => true,
                    'prevention_method' => 'immediate_signature'
                ]);
                return;
            }
            
            // Mark this request as processed immediately
            $this->session->set_userdata($immediate_check_key, time());
            $this->session->mark_as_temp($immediate_check_key, 60); // 1 minute
            
            // Enhanced duplicate prevention with multiple approaches
            $form_token = $this->input->post('form_token');
            $post_data = $this->input->post();
            
            // Create a unique fingerprint for this submission
            $submission_fingerprint = md5(serialize([
                'lead_id' => $lead_id,
                'description' => trim($post_data['description']),
                'staff_id' => get_staff_user_id(),
                'followup_type' => isset($post_data['followup_type']) ? $post_data['followup_type'] : '',
                'next_followup_date' => isset($post_data['next_followup_date']) ? $post_data['next_followup_date'] : ''
            ]));
            
            // Method 1: Form token based prevention
            $session_key = 'lead_note_' . $lead_id . '_' . $form_token;
            $token_duplicate = $form_token && $this->session->userdata($session_key);
            
            // Method 2: Fingerprint based prevention (for rapid submissions)
            $fingerprint_key = 'submission_' . $submission_fingerprint;
            $fingerprint_duplicate = $this->session->userdata($fingerprint_key);
            
            // Method 3: Database check for recent identical notes (within last 30 seconds)
            $this->db->where('leadid', $lead_id);
            $this->db->where('addedfrom', get_staff_user_id());
            $this->db->where('description LIKE', '%' . trim($post_data['description']) . '%');
            $this->db->where('dateadded >', date('Y-m-d H:i:s', time() - 30));
            $recent_note = $this->db->get(db_prefix() . 'lead_notes')->row();
            $database_duplicate = !empty($recent_note);
            
            // Log duplicate detection for debugging
            if ($token_duplicate || $fingerprint_duplicate || $database_duplicate) {
                $debug_info = [
                    'timestamp' => date('Y-m-d H:i:s'),
                    'lead_id' => $lead_id,
                    'staff_id' => get_staff_user_id(),
                    'form_token' => $form_token,
                    'fingerprint' => $submission_fingerprint,
                    'token_duplicate' => $token_duplicate,
                    'fingerprint_duplicate' => $fingerprint_duplicate,
                    'database_duplicate' => $database_duplicate,
                    'description_preview' => substr(trim($post_data['description']), 0, 50)
                ];
                log_message('info', "DUPLICATE FOLLOWUP PREVENTED: " . json_encode($debug_info));
            }
            
            if ($token_duplicate || $fingerprint_duplicate || $database_duplicate) {
                // Form already processed or duplicate detected
                echo json_encode([
                    'success'  => true,
                    'message'  => _l('added_successfully', _l('followup')),
                    'leadView' => $this->_get_lead_data($lead_id),
                    'id'       => $lead_id,
                    'duplicate_prevented' => true,
                    'debug_info' => [
                        'token_duplicate' => $token_duplicate,
                        'fingerprint_duplicate' => $fingerprint_duplicate,
                        'database_duplicate' => $database_duplicate,
                        'fingerprint' => $submission_fingerprint
                    ]
                ]);
                return;
            }
            
            // Mark this form as processed using both methods
            if ($form_token) {
                $this->session->set_userdata($session_key, time());
                $this->session->mark_as_temp($session_key, 300); // 5 minutes
            }
            
            // Mark this fingerprint as processed
            $this->session->set_userdata($fingerprint_key, time());
            $this->session->mark_as_temp($fingerprint_key, 60); // 1 minute for fingerprint
            
            $data = $this->input->post();
            
            // Store follow-up type separately before adding to description
            $followup_type = '';
            if (isset($data['followup_type']) && !empty($data['followup_type'])) {
                $followup_type = trim($data['followup_type']);
                // Add the followup type to the description for display
                $data['description'] = '<strong>Type: ' . htmlspecialchars($followup_type) . '</strong><br>' . $data['description'];
            }
            
            // Handle contact date information
            if (isset($data['contacted_indicator'])) {
                if ($data['contacted_indicator'] == 'yes') {
                    $contacted_date = to_sql_date($data['custom_contact_date'], true);
                    if ($contacted_date == '') {
                        $contacted_date = date('Y-m-d H:i:s');
                    }
                    $data['date_contacted'] = $contacted_date;
                }
                unset($data['contacted_indicator']);
                unset($data['custom_contact_date']);
            }
            
            // Handle next follow-up date
            if (isset($data['next_followup_date']) && !empty($data['next_followup_date'])) {
                // Try to convert the date, but if it fails, keep the original format
                $next_followup = $data['next_followup_date'];
                
                // If it's already in SQL format, use it directly
                if (preg_match('/^\d{4}-\d{2}-\d{2}(\s\d{2}:\d{2}(:\d{2})?)?$/', $next_followup)) {
                    $data['next_followup_date'] = $next_followup;
                } else {
                    // Try to convert using to_sql_date
                    $converted = to_sql_date($next_followup, true);
                    if ($converted && $converted != '' && $converted != $next_followup) {
                        $data['next_followup_date'] = $converted;
                    } else {
                        // If conversion fails, try strtotime
                        $timestamp = strtotime($next_followup);
                        if ($timestamp !== false) {
                            $data['next_followup_date'] = date('Y-m-d H:i:s', $timestamp);
                        } else {
                            // If all fails, remove the field to prevent SQL errors
                            unset($data['next_followup_date']);
                        }
                    }
                }
            } else {
                unset($data['next_followup_date']);
            }
            
            // Add the note to lead_notes table directly with followup_type saved separately
            $insert_data = [
                'description' => nl2br($data['description']),
                'followup_type' => $followup_type, // Save followup type in separate column
                'leadid' => $lead_id,
                'addedfrom' => get_staff_user_id(),
                'dateadded' => date('Y-m-d H:i:s')
            ];
            
            // Add date_contacted if provided
            if (isset($data['date_contacted'])) {
                $insert_data['date_contacted'] = $data['date_contacted'];
            }
            
            // Add next_followup_date if provided (only if column exists)
            if (isset($data['next_followup_date']) && $this->db->field_exists('next_followup_date', db_prefix() . 'lead_notes')) {
                $insert_data['next_followup_date'] = $data['next_followup_date'];
            }
            
            // Add dealer interest if provided (only if column exists)
            if (isset($data['dealer_interest']) && !empty($data['dealer_interest']) && $this->db->field_exists('dealer_interest', db_prefix() . 'lead_notes')) {
                $insert_data['dealer_interest'] = $data['dealer_interest'];
            }
            
            // Add call stats if provided (only if column exists)
            if (isset($data['call_stats']) && !empty($data['call_stats']) && $this->db->field_exists('stats', db_prefix() . 'lead_notes')) {
                $insert_data['stats'] = $data['call_stats'];
            }
            
            // Determine if this is a new call or follow-up call (only if column exists)
            if ($this->db->field_exists('is_new_call', db_prefix() . 'lead_notes')) {
                // Check if followup_type is a call type
                $isCallType = false;
                if (!empty($followup_type)) {
                    $callTypes = ['call', 'phone call', 'whatsapp call', 'phone', 'dial'];
                    $followupLower = strtolower($followup_type);
                    foreach ($callTypes as $callType) {
                        if (strpos($followupLower, $callType) !== false) {
                            $isCallType = true;
                            break;
                        }
                    }
                }
                
                if ($isCallType) {
                    // Check if this lead has been called before
                    $this->db->select('id');
                    $this->db->from(db_prefix() . 'lead_notes');
                    $this->db->where('leadid', $lead_id);
                    $this->db->where('followup_type IS NOT NULL', null, false);
                    
                    // Check if any previous note has a call-type followup
                    $this->db->where('(
                        LOWER(followup_type) LIKE "%call%" OR 
                        LOWER(followup_type) LIKE "%phone%" OR 
                        LOWER(followup_type) LIKE "%dial%"
                    )', null, false);
                    
                    $this->db->limit(1);
                    $previous_call = $this->db->get()->row();
                    
                    // If no previous call found, this is a new call (first time calling this lead)
                    // If previous call exists, this is a follow-up call
                    $insert_data['is_new_call'] = empty($previous_call) ? 1 : 0;
                } else {
                    // Not a call type, set to NULL
                    $insert_data['is_new_call'] = null;
                }
            }
            
            // Insert into lead_notes table
            $this->db->insert(db_prefix() . 'lead_notes', $insert_data);
            $note_id = $this->db->insert_id();
            
            // Check for database errors
            if (!$note_id) {
                $db_error = $this->db->error();
                log_message('error', 'Failed to insert lead note: ' . $db_error['message']);
                log_message('error', 'Insert data: ' . print_r($insert_data, true));
            }
            
            $success = false;
            $message = '';
            
            // Handle attachment if any
            if ($note_id) {
                if (isset($_FILES['file']['name']) && $_FILES['file']['name'] != '') {
                    hooks()->do_action('before_upload_lead_attachment', $lead_id);
                    $attachment = handle_lead_note_attachment($lead_id, $note_id);
                    if ($attachment) {
                        $this->leads_model->log_lead_activity($lead_id, 'not_lead_note_attachment_added');
                    }
                }
                
                // Create automatic reminder if next_followup_date is set
                if (isset($data['next_followup_date']) && !empty($data['next_followup_date'])) {
                    $this->load->model('misc_model');
                    
                    // Prepare reminder data
                    $reminder_data = [
                        'description' => 'Follow-up reminder: ' . strip_tags($data['description']),
                        'date' => $data['next_followup_date'],
                        'rel_id' => $lead_id,
                        'rel_type' => 'lead',
                        'staff' => get_staff_user_id(),
                        'notify_by_email' => 1, // Enable email notification by default
                        'isnotified' => 0
                    ];
                    
                    // Add the reminder
                    $reminder_created = $this->misc_model->add_reminder($reminder_data, $lead_id);
                    
                    if ($reminder_created) {
                        // Log that reminder was created
                        $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_added_followup_reminder');
                    }
                }
                
                // Log the activity ONLY if note was successfully created
                $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_added_followup');
                
                $success = true;
                $message = _l('added_successfully', _l('followup'));
                
                // Add additional message if reminder was created
                if (isset($reminder_created) && $reminder_created) {
                    $message .= ' ' . _l('reminder_created_automatically');
                }
            } else {
                $success = false;
                $message = _l('error_adding_followup');
            }
        }
        
        echo json_encode([
            'success'  => $success,
            'message'  => $message,
            'leadView' => $this->_get_lead_data($lead_id),
            'id'       => $lead_id,
        ]);
    }

    public function update_all_proposal_emails_linked_to_lead($id)
    {
        $success = false;
        $email   = '';
        if ($this->input->post('update')) {
            $this->load->model('proposals_model');

            $this->db->select('email');
            $this->db->where('id', $id);
            $email = $this->db->get(db_prefix() . 'leads')->row()->email;

            $proposals = $this->proposals_model->get('', [
                'rel_type' => 'lead',
                'rel_id'   => $id,
            ]);
            $affected_rows = 0;

            foreach ($proposals as $proposal) {
                $this->db->where('id', $proposal['id']);
                $this->db->update(db_prefix() . 'proposals', [
                    'email' => $email,
                ]);
                if ($this->db->affected_rows() > 0) {
                    $affected_rows++;
                }
            }

            if ($affected_rows > 0) {
                $success = true;
            }
        }

        echo json_encode([
            'success' => $success,
            'message' => _l('proposals_emails_updated', [
                _l('lead_lowercase'),
                $email,
            ]),
        ]);
    }

    public function save_form_data()
    {
        $data = $this->input->post();

        // form data should be always sent to the request and never should be empty
        // this code is added to prevent losing the old form in case any errors
        if (!isset($data['formData']) || isset($data['formData']) && !$data['formData']) {
            echo json_encode([
                'success' => false,
            ]);
            die;
        }

        // If user paste with styling eq from some editor word and the Codeigniter XSS feature remove and apply xss=remove, may break the json.
        $data['formData'] = preg_replace('/=\\\\/m', "=''", $data['formData']);

        $this->db->where('id', $data['id']);
        $this->db->update(db_prefix() . 'web_to_lead', [
            'form_data' => $data['formData'],
        ]);
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'success' => true,
                'message' => _l('updated_successfully', _l('web_to_lead_form')),
            ]);
        } else {
            echo json_encode([
                'success' => false,
            ]);
        }
    }

    public function form($id = '')
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }
        if ($this->input->post()) {
            if ($id == '') {
                $data = $this->input->post();
                $id   = $this->leads_model->add_form($data);
                if ($id) {
                    set_alert('success', _l('added_successfully', _l('web_to_lead_form')));
                    redirect(admin_url('leads/form/' . $id));
                }
            } else {
                $success = $this->leads_model->update_form($id, $this->input->post());
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('web_to_lead_form')));
                }
                redirect(admin_url('leads/form/' . $id));
            }
        }

        $data['formData'] = [];
        $custom_fields    = get_custom_fields('leads', 'type != "link"');

        $cfields       = format_external_form_custom_fields($custom_fields);
        $data['title'] = _l('web_to_lead');

        if ($id != '') {
            $data['form'] = $this->leads_model->get_form([
                'id' => $id,
            ]);
            $data['title']    = $data['form']->name . ' - ' . _l('web_to_lead_form');
            $data['formData'] = $data['form']->form_data;
        }

        $this->load->model('roles_model');
        $data['roles']    = $this->roles_model->get();
        $data['sources']  = $this->leads_model->get_source();
        $data['statuses'] = $this->leads_model->get_status();

        $data['members'] = $this->staff_model->get('', [
            'active'       => 1,
            'is_not_staff' => 0,
        ]);

        $data['languages'] = $this->app->get_available_languages();
        $data['cfields']   = $cfields;

        $db_fields = [];
        $fields    = [
            'name',
            'title',
            'email',
            'phonenumber',
            'lead_value',
            'company',
            'address',
            'city',
            'state',
            'country',
            'zip',
            'description',
            'website',
        ];

        $fields = hooks()->apply_filters('lead_form_available_database_fields', $fields);

        $className = 'form-control';

        foreach ($fields as $f) {
            $_field_object = new stdClass();
            $type          = 'text';
            $subtype       = '';
            if ($f == 'email') {
                $subtype = 'email';
            } elseif ($f == 'description' || $f == 'address') {
                $type = 'textarea';
            } elseif ($f == 'country') {
                $type = 'select';
            }

            if ($f == 'name') {
                $label = _l('lead_add_edit_name');
            } elseif ($f == 'email') {
                $label = _l('lead_add_edit_email');
            } elseif ($f == 'phonenumber') {
                $label = _l('lead_add_edit_phonenumber');
            } elseif ($f == 'lead_value') {
                $label = _l('lead_add_edit_lead_value');
                $type  = 'number';
            } else {
                $label = _l('lead_' . $f);
            }

            $field_array = [
                'subtype'   => $subtype,
                'type'      => $type,
                'label'     => $label,
                'className' => $className,
                'name'      => $f,
            ];

            if ($f == 'country') {
                $field_array['values'] = [];

                $field_array['values'][] = [
                    'label'    => '',
                    'value'    => '',
                    'selected' => false,
                ];

                $countries = get_all_countries();
                foreach ($countries as $country) {
                    $selected = false;
                    if (get_option('customer_default_country') == $country['country_id']) {
                        $selected = true;
                    }
                    array_push($field_array['values'], [
                        'label'    => $country['short_name'],
                        'value'    => (int) $country['country_id'],
                        'selected' => $selected,
                    ]);
                }
            }

            if ($f == 'name') {
                $field_array['required'] = true;
            }

            $_field_object->label    = $label;
            $_field_object->name     = $f;
            $_field_object->fields   = [];
            $_field_object->fields[] = $field_array;
            $db_fields[]             = $_field_object;
        }
        $data['bodyclass'] = 'web-to-lead-form';
        $data['db_fields'] = $db_fields;
        $this->load->view('admin/leads/formbuilder', $data);
    }

    public function forms($id = '')
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }

        if ($this->input->is_ajax_request()) {
            $this->app->get_table_data('web_to_lead');
        }

        $data['title'] = _l('web_to_lead');
        $this->load->view('admin/leads/forms', $data);
    }

    public function delete_form($id)
    {
        if (!is_admin()) {
            access_denied('Web To Lead Access');
        }

        $success = $this->leads_model->delete_form($id);
        if ($success) {
            set_alert('success', _l('deleted', _l('web_to_lead_form')));
        }

        redirect(admin_url('leads/forms'));
    }

    public function sources()
    {
        if (!is_admin()) {
            access_denied('Leads Sources');
        }
        $data['sources'] = $this->leads_model->get_source();
        $data['title']   = 'Leads sources';
        $this->load->view('admin/leads/manage_sources', $data);
    }

    /* Add or update leads sources */
    public function source()
    {
        if (!is_admin() && get_option('staff_members_create_inline_lead_source') == '0') {
            access_denied('Leads Sources');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }

                $id = $this->leads_model->add_source($data);

                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('lead_source')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->leads_model->update_source($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('lead_source')));
                }
            }
        }
    }

    /* Delete leads source */
    public function delete_source($id)
    {
        if (!is_admin()) {
            access_denied('Delete Lead Source');
        }
        if (!$id) {
            redirect(admin_url('leads/sources'));
        }
        $response = $this->leads_model->delete_source($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_source_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('lead_source')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_source_lowercase')));
        }
        redirect(admin_url('leads/sources'));
    }

    // Statuses
    /* View leads statuses */
    public function statuses()
    {
        if (!is_admin()) {
            access_denied('Leads Statuses');
        }
        $data['statuses'] = $this->leads_model->get_status();
        $data['title']    = 'Leads statuses';
        $this->load->view('admin/leads/manage_statuses', $data);
    }

    /* Add or update leads status */
    public function status()
    {
        if (!is_admin() && get_option('staff_members_create_inline_lead_status') == '0') {
            access_denied('Leads Statuses');
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            if (!$this->input->post('id')) {
                $inline = isset($data['inline']);
                if (isset($data['inline'])) {
                    unset($data['inline']);
                }
                $id = $this->leads_model->add_status($data);
                if (!$inline) {
                    if ($id) {
                        set_alert('success', _l('added_successfully', _l('lead_status')));
                    }
                } else {
                    echo json_encode(['success' => $id ? true : false, 'id' => $id]);
                }
            } else {
                $id = $data['id'];
                unset($data['id']);
                $success = $this->leads_model->update_status($data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('lead_status')));
                }
            }
        }
    }

    /* Delete leads status from database */
    public function delete_status($id)
    {
        if (!is_admin()) {
            access_denied('Leads Statuses');
        }
        if (!$id) {
            redirect(admin_url('leads/statuses'));
        }
        $response = $this->leads_model->delete_status($id);
        if (is_array($response) && isset($response['referenced'])) {
            set_alert('warning', _l('is_referenced', _l('lead_status_lowercase')));
        } elseif ($response == true) {
            set_alert('success', _l('deleted', _l('lead_status')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('lead_status_lowercase')));
        }
        redirect(admin_url('leads/statuses'));
    }

    /**
     * Manage Follow-up Types
     * This function handles the page for managing lead follow-up types
     */
    public function manage_followup_types()
    {
        // Check for admin permissions
        if (!is_admin()) {
            access_denied('Leads Follow-up Types Management');
        }

        $this->load->model('misc_model');

        // Handle AJAX requests for DataTables
        if ($this->input->is_ajax_request()) {
            $types = $this->misc_model->get_followup_types();
            
            // Debug log to see what's in the types array
            log_activity('Follow-up Types Debug: ' . json_encode($types));
            
            $output = array(
                'draw' => intval($this->input->post('draw')),
                'recordsTotal' => count($types),
                'recordsFiltered' => count($types),
                'data' => array()
            );

            foreach ($types as $type) {
                $options = '<a href="javascript:void(0)" class="btn btn-default btn-icon" onclick="edit_type(\'' . html_escape($type) . '\')" title="' . _l('edit') . '"><i class="fa fa-pencil-square-o"></i></a> <a href="javascript:void(0)" class="btn btn-danger btn-icon" onclick="delete_type(\'' . html_escape($type) . '\')" title="' . _l('delete') . '"><i class="fa fa-remove"></i></a>';

                $output['data'][] = array(
                    html_escape($type),
                    $options
                );
            }
            
            echo json_encode($output);
            return;
        }

    // Load the view for managing follow-up types
    $data = [];
    $data['title'] = _l('lead_followup_types');
   
    
    $this->load->view('admin/leads/manage_followup_types', $data);
    }

    /**
     * Add a new follow-up type
     */
    public function add_followup_type()
    {
        if (!is_admin()) {
            access_denied('Lead Follow-up Types');
        }

        if ($this->input->post()) {
            $name = trim($this->input->post('name'));
            
            if (empty($name)) {
                set_alert('warning', _l('name_required'));
                redirect(admin_url('leads/manage_followup_types'));
                return;
            }
            
            try {
                $this->load->model('misc_model');
                $types = $this->misc_model->get_followup_types();
                
                if (!in_array($name, $types)) {
                    $types[] = $name;
                    $json = json_encode(array_values($types));
                    
                    $result = update_option('lead_followup_types', $json);
                    
                    if ($result) {
                        set_alert('success', _l('added_successfully', _l('followup_type')));
                    } else {
                        set_alert('danger', _l('error_adding_followup_type'));
                    }
                } else {
                    set_alert('warning', _l('followup_type_exists', $name));
                }
            } catch (Exception $e) {
                log_message('error', 'Error adding follow-up type: ' . $e->getMessage());
                set_alert('danger', _l('error_adding_followup_type'));
            }
            
            redirect(admin_url('leads/manage_followup_types'));
        }
    }

    /**
     * Delete a follow-up type
     * 
     * @param string $name The name of the follow-up type to delete
     */
    /**
     * Delete a follow-up type
     *
     * @param string $name The name of the follow-up type to delete
     */
    public function delete_followup_type($name = '')
    {
        if (!is_admin()) {
            access_denied('Lead Follow-up Types');
        }
        
        if ($name) {
            try {
                $name = urldecode($name);
                $this->load->model('misc_model');
                $types = $this->misc_model->get_followup_types();
                
                $types = array_filter($types, function($type) use ($name) {
                    return $type !== $name;
                });
                
                $result = update_option('lead_followup_types', json_encode(array_values($types)));
                
                if ($result) {
                    set_alert('success', _l('deleted', _l('followup_type')));
                } else {
                    set_alert('danger', _l('error_deleting_followup_type'));
                }
            } catch (Exception $e) {
                log_message('error', 'Error deleting follow-up type: ' . $e->getMessage());
                set_alert('danger', _l('error_deleting_followup_type'));
            }
        } else {
            set_alert('warning', 'No follow-up type name provided');
        }
        
        redirect(admin_url('leads/manage_followup_types'));
    }

    public function email_integration_folders()
    {
        if (!is_admin()) {
            ajax_access_denied('Leads Test Email Integration');
        }

        app_check_imap_open_function();

        $imap = new Imap(
            $this->input->post('email'),
            $this->input->post('password', false),
            $this->input->post('imap_server'),
            $this->input->post('encryption')
        );

        try {
            echo json_encode($imap->getSelectableFolders());
        } catch (ConnectionErrorException $e) {
            echo json_encode([
                'alert_type' => 'warning',
                'message'    => $e->getMessage(),
            ]);
        }
    }

    public function test_email_integration()
    {
        if (!is_admin()) {
            access_denied('Leads Test Email Integration');
        }

        app_check_imap_open_function(admin_url('leads/email_integration'));

        $mail     = $this->leads_model->get_email_integration();
        $password = $mail->password;

        if (false == $this->encryption->decrypt($password)) {
            set_alert('danger', _l('failed_to_decrypt_password'));
            redirect(admin_url('leads/email_integration'));
        }

        $imap = new Imap(
            $mail->email,
            $this->encryption->decrypt($password),
            $mail->imap_server,
            $mail->encryption
        );

        try {
            $connection = $imap->testConnection();

            try {
                $connection->getMailbox($mail->folder);
                set_alert('success', _l('lead_email_connection_ok'));
            } catch (MailboxDoesNotExistException $e) {
                set_alert('danger', str_replace(["\n", 'Mailbox'], ['<br />', 'Folder'], addslashes($e->getMessage())));
            }
        } catch (ConnectionErrorException $e) {
            $error = str_replace("\n", '<br />', addslashes($e->getMessage()));
            set_alert('danger', _l('lead_email_connection_not_ok') . '<br /><br /><b>' . $error . '</b>');
        }

        redirect(admin_url('leads/email_integration'));
    }

    public function email_integration()
    {
        if (!is_admin()) {
            access_denied('Leads Email Intregration');
        }
        if ($this->input->post()) {
            $data             = $this->input->post();
            $data['password'] = $this->input->post('password', false);

            if (isset($data['fakeusernameremembered'])) {
                unset($data['fakeusernameremembered']);
            }
            if (isset($data['fakepasswordremembered'])) {
                unset($data['fakepasswordremembered']);
            }

            $success = $this->leads_model->update_email_integration($data);
            if ($success) {
                set_alert('success', _l('leads_email_integration_updated'));
            }
            redirect(admin_url('leads/email_integration'));
        }
        $data['roles']    = $this->roles_model->get();
        $data['sources']  = $this->leads_model->get_source();
        $data['statuses'] = $this->leads_model->get_status();

        $data['members'] = $this->staff_model->get('', [
            'active'       => 1,
            'is_not_staff' => 0,
        ]);

        $data['title'] = _l('leads_email_integration');
        $data['mail']  = $this->leads_model->get_email_integration();

        $data['bodyclass'] = 'leads-email-integration';
        $this->load->view('admin/leads/email_integration', $data);
    }

    public function change_status_color()
    {
        if ($this->input->post() && is_admin()) {
            $this->leads_model->change_status_color($this->input->post());
        }
    }

    public function import()
    {
        if (!is_admin() && get_option('allow_non_admin_members_to_import_leads') != '1') {
            access_denied('Leads Import');
        }

        $dbFields = $this->db->list_fields(db_prefix() . 'leads');
        array_push($dbFields, 'tags');
        
        // Add followup fields for import
        array_push($dbFields, 'followup_description');
        array_push($dbFields, 'followup_type');
        array_push($dbFields, 'followup_date_contacted');

        $this->load->library('import/import_leads', [], 'import');
        $this->import->setDatabaseFields($dbFields)
        ->setCustomFields(get_custom_fields('leads'));

        if ($this->input->post('download_sample') === 'true') {
            $this->import->downloadSample();
        }

        if ($this->input->post()
            && isset($_FILES['file_csv']['name']) && $_FILES['file_csv']['name'] != '') {
            $this->import->setSimulation($this->input->post('simulate'))
                          ->setTemporaryFileLocation($_FILES['file_csv']['tmp_name'])
                          ->setFilename($_FILES['file_csv']['name'])
                          ->perform();

            $data['total_rows_post'] = $this->import->totalRows();

            if (!$this->import->isSimulation()) {
                set_alert('success', _l('import_total_imported', $this->import->totalImported()));
            }
        }

        $data['statuses'] = $this->leads_model->get_status();
        $data['sources']  = $this->leads_model->get_source();
        $data['members']  = $this->staff_model->get('', ['is_not_staff' => 0, 'active' => 1]);

        $data['title'] = _l('import');
        $this->load->view('admin/leads/import', $data);
    }

    public function validate_unique_field()
    {
        if ($this->input->post()) {

            // First we need to check if the field is the same
            $lead_id = $this->input->post('lead_id');
            $field   = $this->input->post('field');
            $value   = $this->input->post($field);

            if ($lead_id != '') {
                $this->db->select($field);
                $this->db->where('id', $lead_id);
                $row = $this->db->get(db_prefix() . 'leads')->row();
                if ($row->{$field} == $value) {
                    echo json_encode(true);
                    die();
                }
            }

            echo total_rows(db_prefix() . 'leads', [ $field => $value ]) > 0 ? 'false' : 'true';
        }
    }

    public function bulk_action()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        hooks()->do_action('before_do_bulk_action_for_leads');
        $total_deleted = 0;
        if ($this->input->post()) {
            $ids                   = $this->input->post('ids');
            $status                = $this->input->post('status');
            $source                = $this->input->post('source');
            $assigned              = $this->input->post('assigned');
            $visibility            = $this->input->post('visibility');
            $tags                  = $this->input->post('tags');
            $last_contact          = $this->input->post('last_contact');
            $lost                  = $this->input->post('lost');
            $has_permission_delete = staff_can('delete',  'leads');
            if (is_array($ids)) {
                foreach ($ids as $id) {
                    if ($this->input->post('mass_delete')) {
                        if ($has_permission_delete) {
                            if ($this->leads_model->delete($id)) {
                                $total_deleted++;
                            }
                        }
                    } else {
                        if ($status || $source || $assigned || $last_contact || $visibility) {
                            $update = [];
                            if ($status) {
                                // We will use the same function to update the status
                                $this->leads_model->update_lead_status([
                                    'status' => $status,
                                    'leadid' => $id,
                                ]);
                            }
                            if ($source) {
                                $update['source'] = $source;
                            }
                            if ($assigned) {
                                $update['assigned'] = $assigned;
                            }
                            if ($last_contact) {
                                $last_contact          = to_sql_date($last_contact, true);
                                $update['lastcontact'] = $last_contact;
                            }

                            if ($visibility) {
                                if ($visibility == 'public') {
                                    $update['is_public'] = 1;
                                } else {
                                    $update['is_public'] = 0;
                                }
                            }

                            if (count($update) > 0) {
                                $this->db->where('id', $id);
                                $this->db->update(db_prefix() . 'leads', $update);
                            }
                        }
                        if ($tags) {
                            handle_tags_save($tags, $id, 'lead');
                        }
                        if ($lost == 'true') {
                            $this->leads_model->mark_as_lost($id);
                        }
                    }
                }
            }
        }

        if ($this->input->post('mass_delete')) {
            set_alert('success', _l('total_leads_deleted', $total_deleted));
        }
    }    public function download_files($lead_id)
    {
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }

        $files = $this->leads_model->get_lead_attachments($lead_id);

        if (count($files) == 0) {
            redirect(previous_url() ?: $_SERVER['HTTP_REFERER']);
        }

        $path = get_upload_path_by_type('lead') . $lead_id;

        $this->load->library('zip');

        foreach ($files as $file) {
            $this->zip->read_file($path . '/' . $file['file_name']);
        }

        $this->zip->download('files.zip');
        $this->zip->clear_data();
    }

    /**
     * Check if a phone number already exists in the database
     * Returns JSON with lead details if the phone number exists
     */
    public function check_phone_exists()
    {
        if (!is_staff_member()) {
            ajax_access_denied();
        }

        $phone = $this->input->post('phone');
        if (empty($phone)) {
            echo json_encode(['exists' => false]);
            return;
        }

        $exclude_id = $this->input->post('lead_id') ?: null;
        
        // Normalize the phone number: remove non-digits, country codes, and leading zeros
        $phone = $this->normalize_phone_for_duplicate_check($phone);
        
        // Check primary phone number with normalized comparison
        $this->db->select(db_prefix() . 'leads.id, ' . db_prefix() . 'leads.name, ' . db_prefix() . 'leads.email, ' . 
                        db_prefix() . 'leads.phonenumber, ' . db_prefix() . 'leads.company, ' . 
                        db_prefix() . 'leads_status.name as status_name');
        $this->db->join(db_prefix() . 'leads_status', db_prefix() . 'leads_status.id=' . db_prefix() . 'leads.status', 'left');
        $this->db->where('phonenumber IS NOT NULL AND phonenumber != ""');
        
        if ($exclude_id) {
            $this->db->where(db_prefix() . 'leads.id !=', $exclude_id);
        }
        
        $leads = $this->db->get(db_prefix() . 'leads')->result();
        
        foreach ($leads as $lead) {
            $normalized_lead_phone = $this->normalize_phone_for_duplicate_check($lead->phonenumber);
            if ($normalized_lead_phone === $phone && !empty($normalized_lead_phone)) {
                // Found in primary phone
                echo json_encode([
                    'exists' => true,
                    'lead' => [
                        'id' => $lead->id,
                        'name' => $lead->name,
                        'email' => $lead->email,
                        'phone' => $lead->phonenumber,
                        'company' => $lead->company,
                        'status' => $lead->status_name
                    ]
                ]);
                return;
            }
        }
        
        // Check additional phone numbers (stored as JSON in phone_numbers column)
        $this->db->select(db_prefix() . 'leads.id, ' . db_prefix() . 'leads.name, ' . db_prefix() . 'leads.email, ' . 
                        db_prefix() . 'leads.phonenumber, ' . db_prefix() . 'leads.company, ' . db_prefix() . 'leads.phone_numbers, ' .
                        db_prefix() . 'leads_status.name as status_name');
        $this->db->join(db_prefix() . 'leads_status', db_prefix() . 'leads_status.id=' . db_prefix() . 'leads.status', 'left');
        $this->db->where('phone_numbers IS NOT NULL');
        
        if ($exclude_id) {
            $this->db->where(db_prefix() . 'leads.id !=', $exclude_id);
        }
        
        $leads_with_additional = $this->db->get(db_prefix() . 'leads')->result();
        
        foreach ($leads_with_additional as $lead) {
            $additional_numbers = json_decode($lead->phone_numbers, true);
            if (!is_array($additional_numbers)) {
                continue;
            }
            
            foreach ($additional_numbers as $additional_phone) {
                $normalized_additional = $this->normalize_phone_for_duplicate_check($additional_phone);
                if ($normalized_additional === $phone && !empty($normalized_additional)) {
                    // Found in additional phones
                    echo json_encode([
                        'exists' => true,
                        'lead' => [
                            'id' => $lead->id,
                            'name' => $lead->name,
                            'email' => $lead->email,
                            'phone' => $additional_phone . ' (additional)',
                            'company' => $lead->company,
                            'status' => $lead->status_name
                        ]
                    ]);
                    return;
                }
            }
        }
        
        // Phone number doesn't exist
        echo json_encode(['exists' => false]);
    }

    /**
     * Normalize phone number for duplicate checking
     * Extracts the last 10 digits for comparison
     * 
     * @param string $phone The phone number to normalize
     * @return string The last 10 digits of the phone number
     */
    private function normalize_phone_for_duplicate_check($phone)
    {
        if (empty($phone)) {
            return '';
        }
        
        // Remove all non-digit characters
        $phone = preg_replace('/\D/', '', $phone);
        
        // If phone number is less than 10 digits, return as is
        if (strlen($phone) < 10) {
            return $phone;
        }
        
        // Extract the last 10 digits
        $phone = substr($phone, -10);
        
        return $phone;
    }

    /**
     * AI Upload Files - Handle file uploads for AI analysis
     * 
     * @return void Outputs JSON response
     */
    public function ai_upload_files()
    {
        $lead_id = $this->input->post('lead_id');
        
        if (!is_staff_member() || !$this->leads_model->staff_can_access_lead($lead_id)) {
            ajax_access_denied();
        }

        // Check if files were uploaded
        if (empty($_FILES['file'])) {
            echo json_encode([
                'success' => false,
                'message' => 'No file uploaded'
            ]);
            return;
        }

        try {
            // Create upload directory
            $upload_path = $this->_create_ai_upload_directory($lead_id);
            
            // Configure upload
            $config['upload_path'] = $upload_path;
            $config['allowed_types'] = 'jpg|jpeg|png|gif|bmp|pdf|doc|docx|xls|xlsx|txt|csv';
            $config['max_size'] = 10240; // 10MB
            $config['encrypt_name'] = true;
            
            $this->load->library('upload', $config);
            
            if ($this->upload->do_upload('file')) {
                $upload_data = $this->upload->data();
                
                // Success response similar to add_lead_attachment
                echo json_encode([
                    'success' => true,
                    'message' => 'File uploaded successfully',
                    'file_name' => $upload_data['file_name'],
                    'file_path' => $upload_data['full_path'],
                    'id' => $lead_id
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => strip_tags($this->upload->display_errors())
                ]);
            }
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * AI Analysis - Analyze lead with OpenAI Vision API
     * 
     * This method handles the AI-driven lead evaluation using uploaded screenshots
     * and OpenAI's GPT-4o Vision API to predict purchase likelihood
     * 
     * @param int $lead_id Lead ID to analyze
     * @return void Outputs JSON response
     */
    public function ai_analyze($lead_id)
    {
        // Set JSON content type
        header('Content-Type: application/json');
        
        // Security checks
        if (!is_staff_member()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You must be logged in as staff member'
            ]);
            return;
        }
        
        if (!$this->leads_model->staff_can_access_lead($lead_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You cannot access this lead'
            ]);
            return;
        }

        // Check if user has permission for AI analysis (admin or sales_manager only)
        if (!is_admin() && !has_permission('leads', '', 'create')) {
            echo json_encode([
                'success' => false,
                'message' => 'Insufficient permissions for AI analysis'
            ]);
            return;
        }

        // Only allow POST requests
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            echo json_encode([
                'success' => false,
                'message' => 'Only POST requests are allowed'
            ]);
            return;
        }

        $this->load->model('leads_ai_model');

        if ($this->input->post()) {
            try {
                // Get lead data
                $lead = $this->leads_model->get($lead_id);
                if (!$lead) {
                    throw new Exception('Lead not found');
                }
                
                // Get follow-up notes and conversations
                $notes = $this->leads_model->get_lead_note_with_attachment('', $lead_id);
                $activity_log = $this->leads_model->get_lead_activity_log($lead_id);
                
                // Add notes and activity to lead data for AI analysis
                $lead_array = (array)$lead;
                $lead_array['notes'] = $notes;
                $lead_array['activity_log'] = $activity_log;

                // Get uploaded files
                $upload_path = $this->_create_ai_upload_directory($lead_id);
                $uploaded_files = [];
                $file_paths = [];

                // Check for files in the upload directory
                if (is_dir($upload_path)) {
                    $files = glob($upload_path . '*');
                    foreach ($files as $file) {
                        if (is_file($file)) {
                            $uploaded_files[] = basename($file); // Filename for old vision API
                            $file_paths[] = $file; // Full path for new file parsing
                        }
                    }
                }

                // Determine analysis type based on file types
                $has_images = false;
                $has_documents = false;
                
                foreach ($file_paths as $file) {
                    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                    if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif', 'bmp'])) {
                        $has_images = true;
                    } else {
                        $has_documents = true;
                    }
                }

                $analysis_result = null;

                if ($has_images && !$has_documents) {
                    // Pure image analysis (personality/social media) - use enhanced file parsing
                    $analysis_result = $this->leads_ai_model->analyze_lead_with_files(
                        $lead_id,
                        $lead_array,
                        $file_paths
                    );
                    
                    if ($analysis_result['success']) {
                        // Return success response for image analysis
                        echo json_encode([
                            'success' => true,
                            'message' => _l('ai_analysis_completed'),
                            'data' => [
                                'report_id' => $analysis_result['report_id'],
                                'confidence_score' => $analysis_result['analysis']['score'] ?? 0,
                                'verdict' => $analysis_result['analysis']['verdict'] ?? 'Cold',
                                'key_insights' => $analysis_result['analysis']['rationale'] ?? 'Analysis completed',
                                'recommended_actions' => $analysis_result['analysis']['recommendations'] ?? '',
                                'communication_analysis' => $analysis_result['analysis']['communication_analysis'] ?? '',
                                'urgency_level' => $analysis_result['analysis']['confidence_level'] ?? 'Medium',
                                'risk_factors' => $analysis_result['analysis']['risk_factors'] ?? [],
                                'positive_signals' => $analysis_result['analysis']['positive_signals'] ?? [],
                                'files_analyzed' => count($file_paths),
                                'analysis_type' => 'enhanced_conversation_analysis',
                                'created_at' => date('Y-m-d H:i:s')
                            ]
                        ]);
                        return;
                    }
                    
                } else {
                    // Document analysis or mixed content - use enhanced file parsing
                    $analysis_result = $this->leads_ai_model->analyze_lead_with_files(
                        $lead_id,
                        $lead_array,
                        $file_paths
                    );

                    if ($analysis_result['success']) {
                        // Return success response with analysis data
                        echo json_encode([
                            'success' => true,
                            'message' => _l('ai_analysis_completed'),
                            'data' => [
                                'report_id' => $analysis_result['report_id'],
                                'confidence_score' => $analysis_result['analysis']['score'],
                                'verdict' => $analysis_result['analysis']['verdict'],
                                'key_insights' => $analysis_result['analysis']['rationale'],
                                'recommended_actions' => $analysis_result['analysis']['recommendations'],
                                'communication_analysis' => $analysis_result['analysis']['communication_analysis'] ?? '',
                                'urgency_level' => $analysis_result['analysis']['confidence_level'] ?? 'Medium',
                                'risk_factors' => $analysis_result['analysis']['risk_factors'] ?? [],
                                'positive_signals' => $analysis_result['analysis']['positive_signals'] ?? [],
                                'files_analyzed' => count($file_paths),
                                'parsed_files' => array_keys($analysis_result['parsed_files'] ?? []),
                                'analysis_type' => 'enhanced_conversation_analysis',
                                'created_at' => date('Y-m-d H:i:s')
                            ]
                        ]);
                        return;
                    } else {
                        throw new Exception($analysis_result['error']);
                    }
                }

                // If we get here, analysis failed
                throw new Exception('Failed to complete AI analysis');

            } catch (Exception $e) {
                // Log error
                log_message('error', 'AI Analysis Error for Lead ' . $lead_id . ': ' . $e->getMessage());
                
                echo json_encode([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        } else {
            // Invalid request method
            echo json_encode([
                'success' => false,
                'message' => 'Invalid request method'
            ]);
        }
    }

    /**
     * Get AI analysis report for a lead
     * 
     * @param int $lead_id Lead ID
     * @return void Outputs JSON response with latest AI report
     */
    public function get_ai_report($lead_id)
    {
        header('Content-Type: application/json');
        
        if (!is_staff_member()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You must be logged in as staff member'
            ]);
            return;
        }
        
        if (!$this->leads_model->staff_can_access_lead($lead_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You cannot access this lead'
            ]);
            return;
        }

        $this->load->model('leads_ai_model');
        
        try {
            $report = $this->leads_ai_model->get_latest_report($lead_id);
            
            if ($report) {
                // Parse extra data from ai_response field if available
                $extra_data = [];
                if (!empty($report->ai_response)) {
                    $extra_data = json_decode($report->ai_response, true) ?: [];
                }
                
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'id' => $report->id,
                        'confidence_score' => $report->confidence_score ?? $report->score ?? 0,
                        'verdict' => $report->purchase_likelihood ?? $report->verdict ?? 'Cold',
                        'key_insights' => $report->key_insights ?? $report->rationale ?? 'No insights available',
                        'recommended_actions' => $report->recommendations ?? '',
                        'communication_analysis' => $extra_data['communication_analysis'] ?? '',
                        'urgency_level' => $extra_data['confidence_level'] ?? 'Medium',
                        'risk_factors' => $extra_data['risk_factors'] ?? [],
                        'positive_signals' => $extra_data['positive_signals'] ?? [],
                        'analysis_type' => $report->analysis_type ?? 'enhanced_conversation_analysis',
                        'created_at' => $report->created_at,
                        'created_by' => $report->created_by ?? $report->staff_id ?? null,
                        'uploaded_files' => $report->uploaded_files ? json_decode($report->uploaded_files, true) : [],
                        // Legacy fields for backward compatibility
                        'score' => $report->confidence_score ?? $report->score ?? 0,
                        'likelihood_category' => $report->purchase_likelihood ?? $report->verdict ?? 'Cold',
                        'rationale' => $report->key_insights ?? $report->rationale ?? 'No rationale available'
                    ]
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No AI analysis found for this lead'
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Error fetching AI report for lead ' . $lead_id . ': ' . $e->getMessage());
            echo json_encode([
                'success' => false,
                'message' => 'Failed to fetch AI report: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get AI analysis statistics for a specific lead
     * @param int $lead_id Lead ID
     * @return void Outputs JSON response
     */
    public function get_ai_statistics($lead_id)
    {
        header('Content-Type: application/json');
        
        if (!is_staff_member()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You must be logged in as staff member'
            ]);
            return;
        }
        
        if (!$this->leads_model->staff_can_access_lead($lead_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You cannot access this lead'
            ]);
            return;
        }

        $this->load->model('leads_ai_model');
        
        try {
            // Get all AI reports for this lead
            $reports = $this->leads_ai_model->get_reports($lead_id);
            
            $total_analyses = count($reports);
            $avg_score = 0;
            
            if ($total_analyses > 0) {
                $total_score = 0;
                $valid_scores = 0;
                
                foreach ($reports as $report) {
                    // Handle both old and new column names
                    $score = $report->confidence_score ?? $report->score ?? null;
                    if ($score !== null && is_numeric($score)) {
                        $total_score += (float)$score;
                        $valid_scores++;
                    }
                }
                
                if ($valid_scores > 0) {
                    $avg_score = round($total_score / $valid_scores, 1);
                }
            }
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'total_analyses' => $total_analyses,
                    'avg_score' => $avg_score,
                    'last_analysis' => $total_analyses > 0 ? $reports[0]->created_at : null
                ]
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => 'Error fetching AI statistics: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get AI reports history for a specific lead
     * @param int $lead_id Lead ID
     * @return void Outputs JSON response
     */
    public function get_ai_reports_history($lead_id)
    {
        header('Content-Type: application/json');
        
        if (!is_staff_member()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You must be logged in as staff member'
            ]);
            return;
        }
        
        if (!$this->leads_model->staff_can_access_lead($lead_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied: You cannot access this lead'
            ]);
            return;
        }

        try {
            $this->load->model('leads_ai_model');
            
            // Check if table exists in 'crm' database
            $table_exists = $this->db->table_exists(db_prefix() . 'leads_ai_reports');
            if (!$table_exists) {
                echo json_encode([
                    'success' => false,
                    'message' => 'AI reports table does not exist in the crm database. Please run the create_ai_tables.sql script.'
                ]);
                return;
            }
            
            $reports = $this->leads_ai_model->get_reports($lead_id);
            
            if (!is_array($reports)) {
                $reports = [];
            }
            
            $formatted_reports = [];
            foreach ($reports as $report) {
                // Handle different possible column names for backward compatibility
                $summary = '';
                if (isset($report->summary)) {
                    $summary = $report->summary;
                } elseif (isset($report->key_insights)) {
                    $summary = $report->key_insights;
                } elseif (isset($report->rationale)) {
                    $summary = $report->rationale;
                }
                
                $analysis_type = $report->analysis_type ?? 'AI Analysis';
                $confidence_score = $report->confidence_score ?? $report->score ?? 0;
                $verdict = $report->purchase_likelihood ?? $report->verdict ?? 'Unknown';
                
                // Determine badge class for verdict
                $verdict_class = 'default';
                switch (strtolower($verdict)) {
                    case 'hot':
                    case 'high':
                        $verdict_class = 'danger';
                        break;
                    case 'warm':
                    case 'medium':
                        $verdict_class = 'warning';
                        break;
                    case 'cold':
                    case 'low':
                        $verdict_class = 'info';
                        break;
                    case 'qualified':
                        $verdict_class = 'success';
                        break;
                }
                
                // Determine confidence level color
                $confidence_class = 'default';
                if ($confidence_score >= 80) {
                    $confidence_class = 'success';
                } elseif ($confidence_score >= 60) {
                    $confidence_class = 'warning';
                } elseif ($confidence_score >= 40) {
                    $confidence_class = 'info';
                } else {
                    $confidence_class = 'danger';
                }
                
                $formatted_reports[] = [
                    'id' => $report->id,
                    'analysis_type' => ucfirst(str_replace('_', ' ', $analysis_type)),
                    'confidence_score' => $confidence_score,
                    'confidence_class' => $confidence_class,
                    'verdict' => ucfirst($verdict),
                    'verdict_class' => $verdict_class,
                    'summary' => substr($summary, 0, 150) . (strlen($summary) > 150 ? '...' : ''),
                    'full_summary' => $summary,
                    'created_at' => $report->created_at,
                    'formatted_date' => date('M j, Y \a\t g:i A', strtotime($report->created_at)),
                    'time_ago' => time_ago($report->created_at),
                    'status' => $report->status ?? 'completed',
                    'view_url' => admin_url('ai_reports/view/' . $report->id),
                    'created_by' => $report->created_by ?? $report->staff_id ?? null
                ];
            }
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'reports' => $formatted_reports,
                    'total_count' => count($reports)
                ]
            ]);
            
        } catch (Exception $e) {
            // Enhanced error logging
            log_message('error', 'AI Reports History Error for Lead ' . $lead_id . ': ' . $e->getMessage());
            log_message('error', 'Stack trace: ' . $e->getTraceAsString());
            
            echo json_encode([
                'success' => false,
                'message' => 'Error fetching AI reports history: ' . $e->getMessage(),
                'debug_info' => is_admin() ? $e->getTraceAsString() : 'Contact administrator for details'
            ]);
        }
    }

    /**
     * Create AI upload directory for lead
     * 
     * @param int $lead_id Lead ID
     * @return string Upload path
     */
    private function _create_ai_upload_directory($lead_id)
    {
        $upload_path = get_upload_path_by_type('lead_ai') . $lead_id . '/';
        
        // Create directory if it doesn't exist
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0755, true);
        }
        
        return $upload_path;
    }

    /**
     * Debug method to test lead status update functionality
     * Access via: /admin/leads/debug_status_update
     */
    public function debug_status_update()
    {
        if (!is_admin()) {
            access_denied('Debug Access');
        }
        
        header('Content-Type: text/plain');
        
        echo "=== Lead Status Update Debug ===\n\n";
        
        // Get available statuses
        try {
            $statuses = $this->leads_model->get_status();
            echo "✓ Available statuses:\n";
            if (is_array($statuses)) {
                foreach ($statuses as $status) {
                    if (is_object($status)) {
                        echo "  - ID: {$status->id}, Name: {$status->name}\n";
                    } elseif (is_array($status)) {
                        echo "  - ID: {$status['id']}, Name: {$status['name']}\n";
                    }
                }
            } else {
                echo "  No statuses found\n";
            }
        } catch (Exception $e) {
            echo "✗ Error getting statuses: " . $e->getMessage() . "\n";
        }
        
        // Get a sample lead
        try {
            $this->db->limit(1);
            $sample_lead = $this->db->get(db_prefix() . 'leads')->row();
            if ($sample_lead) {
                echo "✓ Sample lead found: ID {$sample_lead->id}, Status: {$sample_lead->status}\n";
                
                // Test the language strings
                echo "\n--- Testing language strings ---\n";
                echo "_l('updated_successfully', 'Lead Status'): " . _l('updated_successfully', 'Lead Status') . "\n";
                echo "_l('something_went_wrong'): " . _l('something_went_wrong') . "\n";
                
                // Test the update function
                $test_data = [
                    'leadid' => $sample_lead->id,
                    'status' => $sample_lead->status
                ];
                
                echo "\n--- Testing status update ---\n";
                echo "Test data: " . json_encode($test_data) . "\n";
                
                $result = $this->leads_model->update_lead_status($test_data);
                echo "Result: " . ($result ? "SUCCESS" : "FAILED") . "\n";
                
                // Show what the AJAX response would look like
                echo "\n--- AJAX Response Preview ---\n";
                $response = [
                    'success' => $result,
                    'message' => $result ? _l('updated_successfully', 'Lead Status') : _l('something_went_wrong')
                ];
                echo "JSON Response: " . json_encode($response) . "\n";
                
            } else {
                echo "✗ No leads found in database\n";
            }
        } catch (Exception $e) {
            echo "✗ Error with sample lead: " . $e->getMessage() . "\n";
        }
        
        echo "\n=== Debug Complete ===\n";
        echo "\nTo test the AJAX response, visit: /test_lead_status_response.html\n";
    }

    /**
     * Test script for followup import functionality
     * Access via: /admin/leads/test_followup_import
     */
    public function test_followup_import()
    {
        if (!is_admin()) {
            access_denied('Test Access');
        }
        
        header('Content-Type: text/plain');
        
        echo "=== Followup Import Feature Test ===\n\n";
        
        // Check followup fields
        $dbFields = $this->db->list_fields(db_prefix() . 'leads');
        array_push($dbFields, 'tags', 'followup_description', 'followup_type', 'followup_date_contacted');
        
        echo "✓ Database fields with followups (" . count($dbFields) . " total):\n";
        $followup_fields = array_filter($dbFields, function($field) {
            return strpos($field, 'followup_') === 0;
        });
        foreach ($followup_fields as $field) {
            echo "  - $field\n";
        }
        
        // Load import library
        $this->load->library('import/import_leads', [], 'import');
        $this->import->setDatabaseFields($dbFields);
        echo "\n✓ Import library loaded successfully\n";
        
        // Check notes table structure
        echo "\n--- Notes Table Structure ---\n";
        try {
            $notes_fields = $this->db->list_fields(db_prefix() . 'notes');
            echo "✓ Notes table fields:\n";
            foreach ($notes_fields as $field) {
                echo "  - $field\n";
            }
        } catch (Exception $e) {
            echo "✗ Error checking notes table: " . $e->getMessage() . "\n";
        }
        
        echo "\n=== Test Complete ===\n";
        echo "\nTo test: Upload /sample_leads_with_followups.csv at /admin/leads/import\n";
    }

    // Removed AJAX endpoint - using fast SQL-based approach instead
}
